<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

require_once 'config/database.php';

$user_id = $_SESSION['user_id'];

// Kullanıcının cüzdanlarını al
try {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ? AND balance > 0 ORDER BY balance DESC");
    $stmt->execute([$user_id]);
    $wallets = $stmt->fetchAll();
} catch (Exception $e) {
    $wallets = [];
}

// Kripto adreslerini al
try {
    $stmt = $db->prepare("SELECT * FROM crypto_addresses WHERE is_active = 1 ORDER BY coin_symbol");
    $stmt->execute();
    $crypto_addresses = $stmt->fetchAll();
} catch (Exception $e) {
    $crypto_addresses = [];
}

// Çekim işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'withdraw_crypto') {
    $coin_symbol = $_POST['coin_symbol'] ?? '';
    $amount = floatval($_POST['amount'] ?? 0);
    $crypto_address_id = intval($_POST['crypto_address'] ?? 0);
    $withdraw_address = $_POST['withdraw_address'] ?? '';
    
    if ($amount <= 0) {
        $error_message = 'Geçerli bir miktar giriniz.';
    } elseif (empty($coin_symbol)) {
        $error_message = 'Para birimi seçiniz.';
    } elseif (empty($withdraw_address)) {
        $error_message = 'Kripto adresinizi giriniz.';
    } else {
        try {
            // Kullanıcının bakiyesini kontrol et
            $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ? AND coin_symbol = ?");
            $stmt->execute([$user_id, $coin_symbol]);
            $wallet = $stmt->fetch();
            
            if (!$wallet || $wallet['available'] < $amount) {
                $error_message = 'Yetersiz bakiye!';
            } else {
                try {
                    $db->beginTransaction();
                    
                    // Bakiyeyi dondur (available'dan frozen'a taşı)
                    $stmt = $db->prepare("UPDATE wallets SET available = available - ?, frozen = frozen + ? WHERE user_id = ? AND coin_symbol = ? AND available >= ?");
                    $stmt->execute([$amount, $amount, $user_id, $coin_symbol, $amount]);
                    
                    // Çekim talebini kaydet
                    $stmt = $db->prepare("INSERT INTO withdrawals (user_id, coin_symbol, amount, bank_account_id, iban, account_holder, status, created_at) VALUES (?, ?, ?, ?, ?, ?, 'pending', NOW())");
                    $stmt->execute([$user_id, $coin_symbol, $amount, NULL, $withdraw_address, 'Crypto Withdrawal']);
                    
                    // İşlem kaydı ekle
                    $stmt = $db->prepare("INSERT INTO transactions (user_id, type, coin_symbol, amount, price, status, created_at) VALUES (?, 'withdraw_pending', ?, ?, 0, 'pending', NOW())");
                    $stmt->execute([$user_id, $coin_symbol, -$amount]);
                    
                    $db->commit();
                    $success_message = 'Kripto çekim talebiniz başarıyla işleme alındı! Tutar kripto adresinize transfer edilecektir.';
                    
                } catch (Exception $e) {
                    $db->rollBack();
                    $error_message = 'Bir hata oluştu: ' . $e->getMessage();
                }
            }
        } catch (Exception $e) {
            $error_message = 'Bir hata oluştu: ' . $e->getMessage();
        }
    }
}

include 'components/header.php';
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kripto Çekim - KriptoCum</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/style.css">
    <script src="assets/js/dropdown.js"></script>
</head>
<body class="bg-gray-50">
    <!-- Main Content -->
    <div class="pt-16 min-h-screen">
        <div class="container mx-auto px-4 py-8">
            <div class="max-w-2xl mx-auto">
            <!-- Header -->
            <div class="mb-8">
                <div class="flex items-center gap-4 mb-4">
                    <button onclick="window.history.back()" class="p-2 hover:bg-gray-100 rounded-lg">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
                        </svg>
                    </button>
                    <div>
                        <h1 class="text-2xl md:text-3xl font-bold text-gray-800 mb-1 md:mb-2">Kripto Çekim Merkezi</h1>
                        <p class="text-sm md:text-base text-gray-600">Kripto varlıklarınızı kendi cüzdanınıza çekin</p>
                    </div>
                </div>
                
                <!-- Badges -->
                <div class="flex gap-2 md:gap-4">
                    <div class="flex items-center gap-2 px-3 py-1.5 bg-green-100 text-green-800 rounded-full text-xs md:text-sm">
                        <svg class="w-4 h-4" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                        </svg>
                        Güvenli
                    </div>
                    <div class="flex items-center gap-2 px-3 py-1.5 bg-blue-100 text-blue-800 rounded-full text-xs md:text-sm">
                        <svg class="w-4 h-4" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                        </svg>
                        Hızlı
                    </div>
                </div>
            </div>

            <!-- Success/Error Messages -->
            <?php if (isset($success_message)): ?>
                <div class="mb-6 p-4 bg-green-100 border border-green-400 text-green-700 rounded-lg">
                    <?php echo htmlspecialchars($success_message); ?>
                </div>
            <?php endif; ?>

            <?php if (isset($error_message)): ?>
                <div class="mb-6 p-4 bg-red-100 border border-red-400 text-red-700 rounded-lg">
                    <?php echo htmlspecialchars($error_message); ?>
                </div>
            <?php endif; ?>

            <!-- Withdrawal Form -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4 md:p-6 mb-8">
                <h2 class="text-lg md:text-xl font-semibold text-gray-800 mb-4 md:mb-6">Kripto Çekim Talebi Oluştur</h2>
                
                <form method="POST" action="">
                    <input type="hidden" name="action" value="withdraw_crypto">
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4 md:gap-6">
                        <!-- Para Birimi -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Çekilecek Kripto Para</label>
                            <select name="coin_symbol" id="coin_symbol" class="w-full px-3 py-2 border border-gray-200 rounded-lg bg-gray-50 focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                                <option value="">Para birimi seçin</option>
                                <?php foreach ($wallets as $wallet): ?>
                                    <option value="<?php echo htmlspecialchars($wallet['coin_symbol']); ?>" data-balance="<?php echo $wallet['available']; ?>">
                                        <?php echo htmlspecialchars($wallet['coin_symbol']); ?> - 
                                        <?php 
                                        if ($wallet['coin_symbol'] === 'TRY') {
                                            echo '₺' . number_format($wallet['available'], 2);
                                        } elseif ($wallet['coin_symbol'] === 'USDT') {
                                            echo number_format($wallet['available'], 2) . ' ' . $wallet['coin_symbol'];
                                        } else {
                                            echo number_format($wallet['available'], 8) . ' ' . $wallet['coin_symbol'];
                                        }
                                        ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <!-- Miktar -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Miktar</label>
                            <input type="number" name="amount" id="amount" step="0.00000001" min="0" class="w-full px-3 py-2 border border-gray-200 rounded-lg bg-gray-50 focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="Çekilecek miktarı girin" required>
                            <div class="mt-1 text-xs md:text-sm text-gray-500">
                                <span id="maxAmount">Maksimum: 0</span>
                            </div>
                            <!-- Yüzde Butonları -->
                            <div class="flex flex-wrap gap-2 mt-2">
                                <button type="button" onclick="setPercentage(25)" class="px-3 py-1.5 bg-gray-100 text-gray-700 rounded-full hover:bg-gray-200 text-xs md:text-sm transition-colors">%25</button>
                                <button type="button" onclick="setPercentage(50)" class="px-3 py-1.5 bg-gray-100 text-gray-700 rounded-full hover:bg-gray-200 text-xs md:text-sm transition-colors">%50</button>
                                <button type="button" onclick="setPercentage(75)" class="px-3 py-1.5 bg-gray-100 text-gray-700 rounded-full hover:bg-gray-200 text-xs md:text-sm transition-colors">%75</button>
                                <button type="button" onclick="setPercentage(100)" class="px-3 py-1.5 bg-gray-100 text-gray-700 rounded-full hover:bg-gray-200 text-xs md:text-sm transition-colors">%100</button>
                            </div>
                        </div>

                        <!-- Kripto Adresi -->
                        <div class="md:col-span-2">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Kendi Kripto Adresiniz</label>
                            <input type="text" name="withdraw_address" id="withdraw_address" class="w-full px-3 py-2 border border-gray-200 rounded-lg bg-gray-50 focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="0x..." required>
                            <p class="mt-1 text-xs md:text-sm text-gray-500">Para gönderilecek kendi kripto adresiniz</p>
                        </div>

                        <!-- Network -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Network (Opsiyonel)</label>
                            <select name="network" id="network" class="w-full px-3 py-2 border border-gray-200 rounded-lg bg-gray-50 focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="">Network seçin</option>
                                <option value="BEP20">BEP20 (Binance Smart Chain)</option>
                                <option value="ERC20">ERC20 (Ethereum)</option>
                                <option value="TRC20">TRC20 (Tron)</option>
                                <option value="BTC">Bitcoin Network</option>
                                <option value="SOL">Solana Network</option>
                                <option value="DOT">Polkadot Network</option>
                            </select>
                        </div>

                        <!-- Açıklama -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Açıklama (Opsiyonel)</label>
                            <textarea name="description" id="description" rows="3" class="w-full px-3 py-2 border border-gray-200 rounded-lg bg-gray-50 focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="Çekim işlemi hakkında notlar..."></textarea>
                        </div>
                    </div>

                    <!-- Submit Button -->
                    <div class="mt-6">
                        <button type="submit" class="w-full px-6 py-3 bg-gradient-to-r from-gray-700 to-gray-600 hover:from-gray-600 hover:to-gray-500 text-white rounded-lg transition-all duration-300 shadow-sm hover:shadow-md font-medium">
                            Kripto Çekim Talebi Oluştur
                        </button>
                    </div>
                </form>
            </div>

            <!-- Important Information -->
            <div class="bg-blue-50 border border-blue-200 rounded-xl p-4 md:p-6">
                <h3 class="text-base md:text-lg font-semibold text-blue-800 mb-3 md:mb-4">Önemli Bilgiler</h3>
                <div class="space-y-2 md:space-y-3 text-xs md:text-sm text-blue-700">
                    <div class="flex items-start gap-2">
                        <svg class="w-5 h-5 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"></path>
                        </svg>
                        <span>Çekim işlemleri için ücret alınmaz.</span>
                    </div>
                    <div class="flex items-start gap-2">
                        <svg class="w-5 h-5 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"></path>
                        </svg>
                        <span>Kendi kripto adresinizi doğru girdiğinizden emin olun. Yanlış adres nedeniyle oluşan kayıplardan sorumlu değiliz.</span>
                    </div>
                </div>
            </div>
            </div>
        </div>
    </div>

    <script>
        // Maksimum miktar güncelleme
        document.getElementById('coin_symbol').addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            const balance = selectedOption.getAttribute('data-balance');
            const maxAmountSpan = document.getElementById('maxAmount');
            
            if (balance) {
                const coinSymbol = selectedOption.value;
                if (coinSymbol === 'TRY') {
                    maxAmountSpan.textContent = 'Maksimum: ₺' + parseFloat(balance).toFixed(2);
                } else {
                    maxAmountSpan.textContent = 'Maksimum: ' + parseFloat(balance).toFixed(8) + ' ' + coinSymbol;
                }
                document.getElementById('amount').max = balance;
            } else {
                maxAmountSpan.textContent = 'Maksimum: 0';
                document.getElementById('amount').max = 0;
            }
        });

        // Yüzde butonları
        function setPercentage(percentage) {
            const coinSelect = document.getElementById('coin_symbol');
            const selectedOption = coinSelect.options[coinSelect.selectedIndex];
            const balance = selectedOption.getAttribute('data-balance');
            
            if (balance) {
                const coinSymbol = coinSelect.value;
                const amount = (parseFloat(balance) * percentage / 100);
                
                // TRY ve USDT için 2 ondalık, diğer coinler için 8 ondalık
                if (coinSymbol === 'TRY' || coinSymbol === 'USDT') {
                    document.getElementById('amount').value = amount.toFixed(2);
                } else {
                    document.getElementById('amount').value = amount.toFixed(8);
                }
            } else {
                alert('Önce para birimi seçin!');
            }
        }
    </script>
<?php include 'components/footer.php'; ?>
</body>
</html>
