<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

require_once 'config/database.php';

$user_id = $_SESSION['user_id'];
// Admin panelinden belirli bir kullanıcıyı görüntüleme desteği
if (isset($_SESSION['admin_id']) && isset($_GET['user_id']) && ctype_digit((string)$_GET['user_id'])) {
    $user_id = (int)$_GET['user_id'];
}

// Kullanıcının cüzdan verilerini veritabanından çek
try {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $wallets = $stmt->fetchAll();
} catch (Exception $e) {
    $wallets = [];
}

// Kripto para fiyatlarını yükle
$crypto_prices = [];
if (file_exists('assets/crypto_prices.json')) {
    $price_data = json_decode(file_get_contents('assets/crypto_prices.json'), true);
    if (isset($price_data['prices'])) {
        $crypto_prices = $price_data['prices'];
    }
}

// Toplam bakiyeleri hesapla
$total_balance = 0;
$available_balance = 0;
$in_orders = 0;
$try_balance = 0;

foreach ($wallets as $wallet) {
    $coin_symbol = $wallet['coin_symbol'];
    $price = 1; // Default price
    
    // TRY'yi toplam bakiyeden hariç tut; ayrı kartta göster
    if ($coin_symbol === 'TRY') {
        $try_balance = isset($wallet['balance']) ? (float)$wallet['balance'] : 0;
        continue;
    }
    
    if (isset($crypto_prices[$coin_symbol])) {
        $price = floatval($crypto_prices[$coin_symbol]['raw_price']);
    }
    
    $total_balance += $wallet['balance'] * $price;
    $available_balance += $wallet['available'] * $price;
    $in_orders += $wallet['frozen'] * $price;
}

include 'components/header.php';
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cüzdan - KriptoCum</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/style.css">
    <script src="assets/js/dropdown.js"></script>
    <style>
        .balance-animation {
            animation: balanceFlash 2s ease-in-out infinite;
        }
        
        @keyframes balanceFlash {
            0% { color: #dc2626; }  /* Kırmızı */
            50% { color: #ffffff; } /* Beyaz */
            100% { color: #dc2626; } /* Kırmızı */
        }
        
        .balance-success {
            animation: balanceSuccess 2s ease-in-out infinite;
        }
        
        @keyframes balanceSuccess {
            0% { color: #16a34a; }  /* Yeşil */
            50% { color: #ffffff; } /* Beyaz */
            100% { color: #16a34a; } /* Yeşil */
        }
        
        .number-animation {
            animation: numberFlash 6s ease-in-out infinite;
        }
        
        @keyframes numberFlash {
            0% { color: #dc2626; }  /* Kırmızı */
            50% { color: #dc2626; } /* Kırmızı */
            50.1% { color: #000000; } /* Siyah */
            100% { color: #000000; } /* Siyah */
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Main Content -->
    <div class="pt-16 min-h-screen">
        <div class="container mx-auto px-4 py-8">
            <!-- Header -->
            <div class="mb-8">
                <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-3">
                    <div>
                        <h1 class="text-2xl md:text-3xl font-bold text-gray-800 mb-1 md:mb-2">Cüzdan</h1>
                        <p class="text-sm md:text-base text-gray-600">Kripto para varlıklarınızı yönetin</p>
                    </div>
                    <div class="flex w-full md:w-auto items-center gap-2 overflow-x-auto md:overflow-visible pb-1.5 md:pb-0 -mx-1 md:mx-0 px-1 md:px-0 whitespace-nowrap">
                        <a href="deposit.php" class="inline-flex items-center justify-center gap-2 whitespace-nowrap rounded-full text-xs md:text-sm font-medium px-3 py-2 md:px-4 md:py-2 bg-gradient-to-r from-gray-700 to-gray-600 hover:from-gray-600 hover:to-gray-500 text-white transition-all duration-300 shadow-sm hover:shadow-md">
                            💰 Para Yatır
                        </a>
                        <a href="convert.php" class="inline-flex items-center justify-center gap-2 whitespace-nowrap rounded-full text-xs md:text-sm font-medium px-3 py-2 md:px-4 md:py-2 bg-gradient-to-r from-gray-700 to-gray-600 hover:from-gray-600 hover:to-gray-500 text-white transition-all duration-300 shadow-sm hover:shadow-md">
                            🔄 Dönüştür
                        </a>
                        <a href="withdraw.php" class="inline-flex items-center justify-center gap-2 whitespace-nowrap rounded-full text-xs md:text-sm font-medium px-3 py-2 md:px-4 md:py-2 bg-gradient-to-r from-gray-700 to-gray-600 hover:from-gray-600 hover:to-gray-500 text-white transition-all duration-300 shadow-sm hover:shadow-md">
                            🏦 Banka Hesabıma Çekim
                        </a>
                        <a href="withdraw-crypto.php" class="inline-flex items-center justify-center gap-2 whitespace-nowrap rounded-full text-xs md:text-sm font-medium px-3 py-2 md:px-4 md:py-2 bg-gradient-to-r from-gray-700 to-gray-600 hover:from-gray-600 hover:to-gray-500 text-white transition-all duration-300 shadow-sm hover:shadow-md">
                            ₿ Kripto ile Çekim
                        </a>
                        <a href="trade.php" class="px-3 py-2 md:px-4 md:py-2 bg-orange-600 text-white rounded-full hover:bg-orange-700 text-xs md:text-sm shadow-sm">
                            📈 İşlem Yap
                        </a>
                    </div>
                </div>
            </div>

            <!-- Wallet Overview -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4 md:gap-6 mb-6 md:mb-8">
                <!-- Total Balance -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4 md:p-6" id="totalBalanceCard">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs md:text-sm font-medium text-gray-600">Toplam Bakiye</p>
                            <p class="text-xl md:text-2xl font-bold text-gray-900 number-animation" id="totalBalance">$<?php echo number_format($total_balance, 2); ?></p>
                        </div>
                        <div class="p-2.5 md:p-3 bg-blue-100 rounded-full">
                            <svg class="w-5 h-5 md:w-6 md:h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <!-- Available Balance -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4 md:p-6" id="availableBalanceCard">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs md:text-sm font-medium text-gray-600">Müsait</p>
                            <p class="text-xl md:text-2xl font-bold text-gray-900 number-animation" id="availableBalance">$<?php echo number_format($available_balance, 2); ?></p>
                        </div>
                        <div class="p-2.5 md:p-3 bg-green-100 rounded-full">
                            <svg class="w-5 h-5 md:w-6 md:h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <!-- In Orders -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4 md:p-6" id="inOrdersCard">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs md:text-sm font-medium text-gray-600">Emirlerde</p>
                            <p class="text-xl md:text-2xl font-bold text-gray-900 number-animation" id="inOrders">$<?php echo number_format($in_orders, 2); ?></p>
                        </div>
                        <div class="p-2.5 md:p-3 bg-yellow-100 rounded-full">
                            <svg class="w-5 h-5 md:w-6 md:h-6 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <!-- TRY Balance -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4 md:p-6" id="tryBalanceCard">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs md:text-sm font-medium text-gray-600">TL Bakiye</p>
                            <p class="text-xl md:text-2xl font-bold text-gray-900 number-animation" id="tryBalance">₺<?php echo number_format($try_balance, 2); ?></p>
                        </div>
                        <div class="p-2.5 md:p-3 bg-red-100 rounded-full">
                            <span class="text-xl md:text-2xl font-bold text-red-600">₺</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Action Buttons -->
            <div class="grid grid-cols-2 md:flex md:flex-wrap gap-3 md:gap-4 mb-6 md:mb-8">
                <button onclick="showDepositModal()" class="px-6 py-3 bg-gradient-to-r from-gray-700 to-gray-600 hover:from-gray-600 hover:to-gray-500 text-white rounded-lg transition-all duration-300 shadow-sm hover:shadow-md flex items-center gap-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                    </svg>
                    Yatır
                </button>
                <button onclick="showWithdrawModal()" class="px-6 py-3 bg-gradient-to-r from-gray-700 to-gray-600 hover:from-gray-600 hover:to-gray-500 text-white rounded-lg transition-all duration-300 shadow-sm hover:shadow-md flex items-center gap-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 12H4m16 0l-4-4m4 4l-4 4"></path>
                    </svg>
                    Çek
                </button>
                <button onclick="showTransferModal()" class="px-6 py-3 bg-gradient-to-r from-gray-700 to-gray-600 hover:from-gray-600 hover:to-gray-500 text-white rounded-lg transition-all duration-300 shadow-sm hover:shadow-md flex items-center gap-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7h12m0 0l-4-4m4 4l-4 4m0 6H4m0 0l4 4m-4-4l4-4"></path>
                    </svg>
                    Aktar
                </button>
                <button onclick="showHistory()" class="px-6 py-3 bg-gradient-to-r from-gray-700 to-gray-600 hover:from-gray-600 hover:to-gray-500 text-white rounded-lg transition-all duration-300 shadow-sm hover:shadow-md flex items-center gap-2">
                    <svg class="w-5 h-5" fill ="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                    Geçmiş
                </button>
            </div>

            <!-- Assets Table (Desktop) -->
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden hidden md:block">
                <div class="p-6 border-b border-gray-200">
                    <div class="flex items-center justify-between">
                        <h2 class="text-xl font-bold text-gray-800">Varlıklar</h2>
                        <div class="flex items-center gap-4">
                            <input type="text" placeholder="🔍 Varlık ara..." class="px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" id="searchInput">
                            <button onclick="showHideZero()" class="px-4 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50">
                                <span id="hideZeroText">Sıfırları Gizle</span>
                            </button>
                        </div>
                    </div>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Varlık</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Toplam</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Müsait</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Emirlerde</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">USD Değeri</th>
                                <th class="px-6 py-3 text-center text-xs font-semibold text-gray-600 uppercase">İşlemler</th>
                            </tr>
                        </thead>
                        <tbody id="assetsTableBody" class="bg-white divide-y divide-gray-200">
                            <!-- Assets will be loaded here -->
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Assets (Mobile Cards) -->
            <div class="md:hidden">
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
                    <div class="p-4 border-b border-gray-200 sticky top-14 bg-white z-10">
                        <div class="flex items-center justify-between gap-2">
                            <h2 class="text-lg font-bold text-gray-800">Varlıklar</h2>
                            <button onclick="showHideZero()" class="px-3 py-1.5 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 whitespace-nowrap">
                                <span id="hideZeroTextMobile">Sıfırları Gizle</span>
                            </button>
                        </div>
                        <div class="mt-3">
                            <input type="text" placeholder="🔍 Varlık ara..." class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" id="searchInputMobile">
                        </div>
                    </div>
                    <div id="mobileAssetsList" class="divide-y divide-gray-100"></div>
                </div>
            </div>
        </div>
    </div>

    <!-- Deposit Modal -->
    <div id="depositModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg max-w-md w-full p-6">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-semibold text-gray-800">Yatır</h3>
                    <button onclick="hideDepositModal()" class="text-gray-400 hover:text-gray-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Varlık Seç</label>
                        <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="BTC">Bitcoin (BTC)</option>
                            <option value="ETH">Ethereum (ETH)</option>
                            <option value="USDT">Tether (USDT)</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Yatırma Adresi</label>
                        <div class="flex">
                            <input type="text" readonly value="1A1zP1eP5QGefi2DMPTfTL5SLmv7DivfNa" class="flex-1 px-3 py-2 border border-gray-300 rounded-l-lg bg-gray-50">
                            <button class="px-3 py-2 bg-blue-600 text-white rounded-r-lg hover:bg-blue-700">Kopyala</button>
                        </div>
                    </div>
                    <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-3">
                        <p class="text-sm text-yellow-800">⚠️ Sadece BTC bu adrese gönderin. Diğer varlıklar göndermek kalıcı kayba neden olabilir.</p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Withdraw Modal -->
    <div id="withdrawModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg max-w-md w-full p-6">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-semibold text-gray-800">Para Çek</h3>
                    <button onclick="hideWithdrawModal()" class="text-gray-400 hover:text-gray-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Varlık Seç</label>
                        <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="BTC">Bitcoin (BTC)</option>
                            <option value="ETH">Ethereum (ETH)</option>
                            <option value="USDT">Tether (USDT)</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Çekme Adresi</label>
                        <input type="text" placeholder="Çekme adresini girin" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Miktar</label>
                        <input type="number" placeholder="0.00" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    <button class="w-full bg-red-600 text-white py-2 rounded-lg hover:bg-red-700">Çek</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Transfer Modal -->
    <div id="transferModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg max-w-md w-full p-6">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-semibold text-gray-800">Transfer</h3>
                    <button onclick="hideTransferModal()" class="text-gray-400 hover:text-gray-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">From</label>
                        <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="spot">Spot Wallet</option>
                            <option value="futures">Futures Wallet</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">To</label>
                        <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="futures">Futures Wallet</option>
                            <option value="spot">Spot Wallet</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Asset</label>
                        <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="USDT">USDT</option>
                            <option value="BTC">BTC</option>
                            <option value="ETH">ETH</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Amount</label>
                        <input type="number" placeholder="0.00" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    <button class="w-full bg-green-600 text-white py-2 rounded-lg hover:bg-green-700">Transfer</button>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Veritabanından gelen cüzdan verileri
        let walletsData = <?php echo json_encode($wallets); ?>;
        const cryptoPrices = <?php echo json_encode($crypto_prices); ?>;
        
        let hideZero = false;
        let assetsData = [];

        function loadAssets() {
            // Veritabanından gelen verileri işle
            assetsData = walletsData.map(wallet => {
                const coin_symbol = wallet.coin_symbol;
                let price = 1;
                
                if (cryptoPrices[coin_symbol]) {
                    price = parseFloat(cryptoPrices[coin_symbol].raw_price);
                } else if (coin_symbol === 'TRY') {
                    price = 0.033; // 1 TRY = 0.033 USD
                }
                
                return {
                    symbol: coin_symbol,
                    name: getCoinName(coin_symbol),
                    total: parseFloat(wallet.balance),
                    available: parseFloat(wallet.available),
                    inOrders: parseFloat(wallet.frozen),
                    price: price
                };
            });
            
            updateAssetsTable();
            updateWalletOverview();
        }

        
        // Bakiye güncelleme animasyonu
        function animateBalanceUpdate() {
            // Sayısal değerlere ekstra animasyon ekle
            const numberElements = document.querySelectorAll('.number-animation');
            numberElements.forEach(element => {
                element.style.animation = 'numberFlash 2s ease-in-out 1';
            });
        }
        
        // Başarılı işlem animasyonu
        function animateSuccess() {
            // Sayısal değerlere başarı animasyonu ekle
            const numberElements = document.querySelectorAll('.number-animation');
            numberElements.forEach(element => {
                element.style.animation = 'balanceSuccess 2s ease-in-out 1';
            });
        }
        

        function getCoinName(symbol) {
            const names = {
                'TRY': 'Türk Lirası',
                'BTC': 'Bitcoin',
                'ETH': 'Ethereum',
                'USDT': 'Tether',
                'USDC': 'USD Coin',
                'SOL': 'Solana',
                'XRP': 'Ripple',
                'ADA': 'Cardano',
                'DOT': 'Polkadot',
                'MATIC': 'Polygon',
                'AVAX': 'Avalanche'
            };
            return names[symbol] || symbol;
        }

        function formatBalance(amount, symbol) {
            if (amount === 0) return '0';
            
            // TRY için özel format
            if (symbol === 'TRY') {
                return amount.toFixed(2);
            }
            
            // USDT için özel format (kısaltma yok)
            if (symbol === 'USDT') {
                return amount.toFixed(2);
            }
            
            // Diğer coinler için akıllı format
            if (amount >= 1000000) {
                return (amount / 1000000).toFixed(2) + 'M';
            } else if (amount >= 1000) {
                return (amount / 1000).toFixed(2) + 'K';
            } else if (amount >= 1) {
                return amount.toFixed(4);
            } else if (amount >= 0.01) {
                return amount.toFixed(6);
            } else {
                return amount.toFixed(8);
            }
        }

        function updateAssetsTable() {
            const tbody = document.getElementById('assetsTableBody');
            if (tbody) tbody.innerHTML = '';
            const mobileList = document.getElementById('mobileAssetsList');
            if (mobileList) mobileList.innerHTML = '';

            // Search filters
            const qDesktop = (document.getElementById('searchInput')?.value || '').toLowerCase().trim();
            const qMobile = (document.getElementById('searchInputMobile')?.value || '').toLowerCase().trim();
            const q = qMobile || qDesktop;

            let filteredAssets = hideZero ? assetsData.filter(asset => asset.total > 0) : assetsData;
            if (q) {
                filteredAssets = filteredAssets.filter(a => a.symbol.toLowerCase().includes(q) || a.name.toLowerCase().includes(q));
            }
            
            // Sıralama: Dolu olanlar üstte, boş olanlar altta
            filteredAssets.sort((a, b) => {
                // Önce bakiye durumuna göre sırala (dolu > boş)
                if (a.total > 0 && b.total === 0) return -1;
                if (a.total === 0 && b.total > 0) return 1;
                
                // İkisi de dolu veya ikisi de boşsa, bakiye miktarına göre sırala (büyük > küçük)
                return b.total - a.total;
            });

            filteredAssets.forEach(asset => {
                const usdValue = asset.total * asset.price;
                // Desktop row
                if (tbody) {
                const tr = document.createElement('tr');
                tr.className = 'hover:bg-gray-50';
                
                // Admin panelinden gelen coin görselini kullan
                let coinImage = '';
                if (cryptoPrices[asset.symbol] && cryptoPrices[asset.symbol].image) {
                    coinImage = cryptoPrices[asset.symbol].image;
                } else if (asset.symbol === 'TRY') {
                    coinImage = 'assets/kriptologolari/try.png';
                } else {
                    coinImage = `images/coin-logos/${asset.symbol.toLowerCase()}.svg`;
                }
                
                tr.innerHTML = `
                    <td class="px-6 py-4">
                        <div class="flex items-center gap-3">
                            <div class="w-8 h-8 rounded-full bg-gray-100 flex items-center justify-center overflow-hidden">
                                <img src="${coinImage}" 
                                     alt="${asset.symbol}" 
                                     class="w-full h-full object-cover"
                                     onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                <div class="w-full h-full bg-gray-100 flex items-center justify-center" style="display: none;">
                                    <span class="text-xs font-semibold">${asset.symbol === 'TRY' ? '₺' : asset.symbol.charAt(0)}</span>
                                </div>
                            </div>
                            <div>
                                <div class="font-semibold text-gray-800">${asset.symbol}</div>
                                <div class="text-xs text-gray-500">${asset.name}</div>
                            </div>
                        </div>
                    </td>
                    <td class="px-6 py-4 text-right text-gray-800 font-mono">
                        ${asset.symbol === 'TRY' ? '₺' + formatBalance(asset.total, asset.symbol) : formatBalance(asset.total, asset.symbol)}
                    </td>
                    <td class="px-6 py-4 text-right text-gray-800 font-mono">
                        ${asset.symbol === 'TRY' ? '₺' + formatBalance(asset.available, asset.symbol) : formatBalance(asset.available, asset.symbol)}
                    </td>
                    <td class="px-6 py-4 text-right text-gray-800 font-mono">
                        ${asset.symbol === 'TRY' ? '₺' + formatBalance(asset.inOrders, asset.symbol) : formatBalance(asset.inOrders, asset.symbol)}
                    </td>
                    <td class="px-6 py-4 text-right text-gray-800 font-semibold number-animation">
                        ${'$' + usdValue.toFixed(2)}
                    </td>
                    <td class="px-6 py-4 text-center">
                        <div class="flex justify-center gap-2">
                            <a href="trade.php" target="_blank" class="text-blue-600 hover:text-blue-700 text-sm">İşlem</a>
                            <button onclick="depositAsset('${asset.symbol}')" class="text-green-600 hover:text-green-700 text-sm">Yatır</button>
                            <button onclick="withdrawAsset('${asset.symbol}')" class="text-red-600 hover:text-red-700 text-sm">Çek</button>
                        </div>
                    </td>
                `;
                tbody.appendChild(tr);
                }

                // Mobile card
                if (mobileList) {
                    const item = document.createElement('div');
                    item.className = 'p-4';
                    item.innerHTML = `
                        <div class="flex items-center gap-3">
                            <div class="w-10 h-10 rounded-full bg-gray-100 flex items-center justify-center overflow-hidden flex-shrink-0">
                                <img src="${coinImage}" alt="${asset.symbol}" class="w-full h-full object-cover" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                <div class="w-full h-full bg-gray-100 flex items-center justify-center" style="display: none;">
                                    <span class="text-sm font-semibold">${asset.symbol === 'TRY' ? '₺' : asset.symbol.charAt(0)}</span>
                                </div>
                            </div>
                            <div class="min-w-0 flex-1">
                                <div class="flex items-center justify-between">
                                    <div class="font-semibold text-gray-800">${asset.symbol} <span class="text-xs text-gray-500 ml-1">${asset.name}</span></div>
                                    <div class="text-right text-gray-800 font-semibold number-animation">${'$' + usdValue.toFixed(2)}</div>
                                </div>
                                <div class="mt-1 grid grid-cols-3 gap-2 text-xs text-gray-600">
                                    <div><span class="text-gray-500">Toplam:</span> ${asset.symbol === 'TRY' ? '₺' + formatBalance(asset.total, asset.symbol) : formatBalance(asset.total, asset.symbol)}</div>
                                    <div><span class="text-gray-500">Müsait:</span> ${asset.symbol === 'TRY' ? '₺' + formatBalance(asset.available, asset.symbol) : formatBalance(asset.available, asset.symbol)}</div>
                                    <div><span class="text-gray-500">Emirde:</span> ${asset.symbol === 'TRY' ? '₺' + formatBalance(asset.inOrders, asset.symbol) : formatBalance(asset.inOrders, asset.symbol)}</div>
                                </div>
                                <div class="mt-3 flex gap-2">
                                    <a href="trade.php" target="_blank" class="px-3 py-1.5 rounded-md border text-xs">İşlem</a>
                                    <button onclick="depositAsset('${asset.symbol}')" class="px-3 py-1.5 rounded-md border text-xs">Yatır</button>
                                    <button onclick="withdrawAsset('${asset.symbol}')" class="px-3 py-1.5 rounded-md border text-xs">Çek</button>
                                </div>
                            </div>
                        </div>
                    `;
                    mobileList.appendChild(item);
                }
            });
        }

        // Live search handlers
        document.addEventListener('input', function(e) {
            if (e.target && (e.target.id === 'searchInput' || e.target.id === 'searchInputMobile')) {
                updateAssetsTable();
            }
        });

        function updateWalletOverview() {
            let totalBalance = 0;
            let availableBalance = 0;
            let inOrders = 0;
            let tryBalance = 0;

            assetsData.forEach(asset => {
                // TRY bakiyesini ayrı hesapla (toplam bakiyeye dahil etme)
                if (asset.symbol === 'TRY') {
                    tryBalance = asset.total;
                    return; // TRY'yi atla
                }

                const usdValue = asset.total * asset.price;
                const availableValue = asset.available * asset.price;
                const ordersValue = asset.inOrders * asset.price;

                totalBalance += usdValue;
                availableBalance += availableValue;
                inOrders += ordersValue;
            });

            document.getElementById('totalBalance').textContent = `$${totalBalance.toFixed(2)}`;
            document.getElementById('availableBalance').textContent = `$${availableBalance.toFixed(2)}`;
            document.getElementById('inOrders').textContent = `$${inOrders.toFixed(2)}`;
            document.getElementById('tryBalance').textContent = `₺${tryBalance.toFixed(2)}`;
        }
        
        
        // Gerçek zamanlı güncelleme başlat
        function startRealTimeUpdates() {
            // Her 60 saniyede bir sayfayı yenile
            setInterval(() => {
                console.log('Otomatik güncelleme...');
                location.reload();
            }, 60000);
        }
        
        // Trade sayfasından gelen bildirimleri dinle
        window.addEventListener('message', function(event) {
            if (event.data.type === 'WALLET_UPDATE') {
                console.log('Wallet güncelleme bildirimi alındı');
                
                // Animasyon ile güncelle
                animateSuccess();
                
                // Sayfayı yenile
                setTimeout(() => {
                    location.reload();
                }, 2000);
            }
        });
        
        function showHideZero() {
            hideZero = !hideZero;
            const t = hideZero ? 'Sıfırları Göster' : 'Sıfırları Gizle';
            const desktop = document.getElementById('hideZeroText');
            const mobile = document.getElementById('hideZeroTextMobile');
            if (desktop) desktop.textContent = t;
            if (mobile) mobile.textContent = t;
            updateAssetsTable();
        }

        // Sayfa yüklendiğinde başlat
        document.addEventListener('DOMContentLoaded', function() {
            loadAssets();
            startRealTimeUpdates();
        });

        function showDepositModal() {
            document.getElementById('depositModal').classList.remove('hidden');
        }

        function hideDepositModal() {
            document.getElementById('depositModal').classList.add('hidden');
        }

        function showWithdrawModal() {
            document.getElementById('withdrawModal').classList.remove('hidden');
        }

        function hideWithdrawModal() {
            document.getElementById('withdrawModal').classList.add('hidden');
        }

        function showTransferModal() {
            document.getElementById('transferModal').classList.remove('hidden');
        }

        function hideTransferModal() {
            document.getElementById('transferModal').classList.add('hidden');
        }

        function showHistory() {
            window.location.href = 'wallet-history.php';
        }

        function depositAsset(symbol) {
            // Trade sayfasına yönlendir
            window.open(`trade.php?symbol=${symbol}`, '_blank');
        }

        function withdrawAsset(symbol) {
            // Trade sayfasına yönlendir
            window.open(`trade.php?symbol=${symbol}`, '_blank');
        }
    </script>

    <?php include 'components/footer.php'; ?>
</body>
</html>


