<?php
session_start();
require_once 'config/database.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

$user_id = $_SESSION['user_id'];
$db = getDB();

// Get user information
$stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$user_id]);
$user = $stmt->fetch();

if (!$user) {
    header('Location: login.php');
    exit();
}

// Kişisel Bilgiler Güncelleme
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_personal') {
    $first_name = isset($_POST['first_name']) ? trim($_POST['first_name']) : '';
    $last_name = isset($_POST['last_name']) ? trim($_POST['last_name']) : '';
    $phone = isset($_POST['phone']) ? trim($_POST['phone']) : null;
    $country = isset($_POST['country']) ? trim($_POST['country']) : null;

    try {
        $upd = $db->prepare("UPDATE users SET first_name = ?, last_name = ?, phone = ?, country = ? WHERE id = ?");
        $upd->execute([$first_name, $last_name, $phone, $country, $user_id]);
        // Veriyi tazele
        $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch();
        $_SESSION['user_name'] = $user['first_name'] . ' ' . $user['last_name'];
        $update_success = true;
    } catch (Exception $e) {
        $update_error = 'Bilgiler güncellenemedi.';
    }
}
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profil - KriptoCum</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="assets/style.css">
</head>
<body class="bg-gray-50">
    <?php include 'components/header.php'; ?>

    <div class="min-h-screen">
        <div class="flex min-h-screen bg-white w-full">
            <!-- Sidebar -->
            <div class="hidden lg:block">
                <div class="w-full max-w-[280px] bg-white border-r border-gray-200 min-h-screen flex flex-col">
                    
                    <!-- Back Button -->
                    <div class="p-3">
                        <a class="flex items-center gap-2 px-2 py-1.5 rounded-md text-xs font-medium text-gray-600 hover:text-gray-900 hover:bg-gray-100 w-fit" href="index.php">
                            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-3.5 w-3.5">
                                <path d="m12 19-7-7 7-7"></path>
                                <path d="M19 12H5"></path>
                            </svg>
                            <span>Ana Sayfa</span>
                        </a>
                    </div>
                    
                    <!-- User Profile Section -->
                    <div class="p-4 lg:p-6 flex flex-col items-center border-b border-gray-200">
                        <div class="relative w-20 lg:w-24 h-20 lg:h-24 mb-4">
                            <span class="relative flex w-full h-full overflow-hidden rounded-full border-2 border-gray-200">
                                <?php 
                                $defaultAvatar = "data:image/svg+xml;utf8,<svg xmlns='http://www.w3.org/2000/svg' width='80' height='80'><rect width='100%' height='100%' fill='%23e5e7eb'/><text x='50%' y='52%' dominant-baseline='middle' text-anchor='middle' fill='%239ca3af' font-family='Arial' font-size='28'>👤</text></svg>";
                                $avatarSrc = !empty($user['profile_image']) ? htmlspecialchars($user['profile_image']) : $defaultAvatar;
                                ?>
                                <img id="sidebar-avatar" class="w-full h-full object-cover" alt="<?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?>" src="<?php echo $avatarSrc; ?>">
                            </span>
                        </div>
                        <div class="text-center">
                            <h2 class="text-xl lg:text-2xl font-bold text-gray-900"><?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?></h2>
                            <p class="text-gray-600 text-sm">@<?php echo htmlspecialchars(explode('@', $user['email'])[0]); ?></p>
                        </div>
                        <div class="mt-4 flex items-center">
                            <span class="bg-blue-100 text-blue-800 text-xs font-medium px-3 py-1.5 rounded-full flex items-center">
                                <span class="w-2 h-2 bg-blue-500 rounded-full mr-1.5"></span>
                                Seviye 0 Doğrulandı
                            </span>
                        </div>
                        <div class="w-full mt-4 lg:mt-6">
                            <div class="flex justify-between text-sm mb-2">
                                <span class="text-gray-600">Profil Tamamlama</span>
                                <span class="font-medium text-gray-900">30%</span>
                            </div>
                            <div class="w-full bg-gray-200 rounded-full h-3 overflow-hidden">
                                <div class="h-full bg-blue-500 rounded-full transition-all duration-700 ease-out" style="width: 30%;"></div>
                            </div>
                            <div class="mt-4 flex flex-col lg:flex-row lg:justify-between lg:items-center gap-2">
                                <span class="text-xs text-gray-600">Üyelik başlangıcı: <?php echo date('d.m.Y', strtotime($user['created_at'])); ?></span>
                                <a class="text-xs text-blue-600 hover:text-blue-800 flex items-center justify-center lg:justify-start" href="#personal">
                                    Komple Profil
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-3 w-3 ml-0.5">
                                        <path d="m9 18 6-6-6-6"></path>
                                    </svg>
                                </a>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Navigation -->
                    <nav class="p-4 flex-1 overflow-y-auto">
                        <ul class="space-y-1">
                            <li>
                                <a class="flex items-center gap-3 px-3 py-2.5 rounded-lg text-sm font-medium bg-blue-50 text-blue-700 cursor-pointer" onclick="showTab('dashboard')">
                                    <div class="p-1.5 rounded-md bg-blue-100">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5">
                                            <line x1="18" x2="18" y1="20" y2="10"></line>
                                            <line x1="12" x2="12" y1="20" y2="4"></line>
                                            <line x1="6" x2="6" y1="20" y2="14"></line>
                                        </svg>
                                    </div>
                                    <span>Kontrol Paneli</span>
                                </a>
                            </li>
                            <li>
                                <a class="flex items-center gap-3 px-3 py-2.5 rounded-lg text-sm font-medium text-gray-600 hover:text-gray-900 hover:bg-gray-100 cursor-pointer" onclick="showTab('personal')">
                                    <div class="p-1.5 rounded-md bg-gray-100">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5">
                                            <path d="M19 21v-2a4 4 0 0 0-4-4H9a4 4 0 0 0-4 4v2"></path>
                                            <circle cx="12" cy="7" r="4"></circle>
                                        </svg>
                                    </div>
                                    <span>Kişisel Bilgiler</span>
                                </a>
                            </li>
                            <li>
                                <a class="flex items-center gap-3 px-3 py-2.5 rounded-lg text-sm font-medium text-gray-600 hover:text-gray-900 hover:bg-gray-100 cursor-pointer" onclick="showTab('security')">
                                    <div class="p-1.5 rounded-md bg-gray-100">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5">
                                            <path d="M20 13c0 5-3.5 7.5-7.66 8.95a1 1 0 0 1-.67-.01C7.5 20.5 4 18 4 13V6a1 1 0 0 1 1-1c2 0 4.5-1.2 6.24-2.72a1.17 1.17 0 0 1 1.52 0C14.51 3.81 17 5 19 5a1 1 0 0 1 1 1z"></path>
                                        </svg>
                                    </div>
                                    <span>Güvenlik</span>
                                </a>
                            </li>
                            <li>
                                <a class="flex items-center gap-3 px-3 py-2.5 rounded-lg text-sm font-medium text-gray-600 hover:text-gray-900 hover:bg-gray-100 cursor-pointer" onclick="showTab('notifications')">
                                    <div class="p-1.5 rounded-md bg-gray-100">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5">
                                            <path d="M10.268 21a2 2 0 0 0 3.464 0"></path>
                                            <path d="M3.262 15.326A1 1 0 0 0 4 17h16a1 1 0 0 0 .74-1.673C19.41 13.956 18 12.499 18 8A6 6 0 0 0 6 8c0 4.499-1.411 5.956-2.738 7.326"></path>
                                        </svg>
                                    </div>
                                    <span>Bildirimler</span>
                                </a>
                            </li>
                            <li>
                                <a class="flex items-center gap-3 px-3 py-2.5 rounded-lg text-sm font-medium text-gray-600 hover:text-gray-900 hover:bg-gray-100" href="wallet.php">
                                    <div class="p-1.5 rounded-md bg-gray-100">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5">
                                            <path d="M19 7V4a1 1 0 0 0-1-1H5a2 2 0 0 0 0 4h15a1 1 0 0 1 1 1v4h-3a2 2 0 0 0 0 4h3a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1"></path>
                                            <path d="M3 5v14a2 2 0 0 0 2 2h15a1 1 0 0 0 1-1v-4"></path>
                                        </svg>
                                    </div>
                                    <span>Cüzdan</span>
                                </a>
                            </li>
                            <li>
                                <a class="flex items-center gap-3 px-3 py-2.5 rounded-lg text-sm font-medium text-gray-600 hover:text-gray-900 hover:bg-gray-100" href="api-management.php">
                                    <div class="p-1.5 rounded-md bg-gray-100">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5">
                                            <path d="m15.5 7.5 2.3 2.3a1 1 0 0 0 1.4 0l2.1-2.1a1 1 0 0 0 0-1.4L19 4"></path>
                                            <path d="m21 2-9.6 9.6"></path>
                                            <circle cx="7.5" cy="15.5" r="5.5"></circle>
                                        </svg>
                                    </div>
                                    <span>API Anahtarları</span>
                                </a>
                            </li>
                        </ul>
                        
                        <!-- Security Tips -->
                        <div class="mt-8 p-4 rounded-lg border border-gray-200 bg-gray-50">
                            <div class="flex items-start space-x-3">
                                <div class="p-2 rounded-full bg-blue-100 flex-shrink-0">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5 text-blue-600">
                                        <path d="M20 13c0 5-3.5 7.5-7.66 8.95a1 1 0 0 1-.67-.01C7.5 20.5 4 18 4 13V6a1 1 0 0 1 1-1c2 0 4.5-1.2 6.24-2.72a1.17 1.17 0 0 1 1.52 0C14.51 3.81 17 5 19 5a1 1 0 0 1 1 1z"></path>
                                    </svg>
                                </div>
                                <div class="min-w-0 flex-1">
                                    <h4 class="text-sm font-medium text-gray-900">Güvenlik İpuçları</h4>
                                    <p class="text-xs text-gray-600 mt-1">Gelişmiş hesap koruması için 2FA'yı etkinleştirin.</p>
                                    <a class="text-xs text-blue-600 hover:underline mt-2 inline-block" href="#security">Güvenlik Ayarları</a>
                                </div>
                            </div>
                        </div>
                    </nav>
                </div>
            </div>
            
            <!-- Mobile Header -->
            <div class="lg:hidden fixed top-0 left-0 right-0 z-50 bg-white border-b border-gray-200 p-4">
                <div class="flex items-center justify-between">
                    <h1 class="text-lg font-semibold">Profil</h1>
                    <button class="inline-flex items-center justify-center h-9 rounded-md px-3">
                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5">
                            <line x1="4" x2="20" y1="12" y2="12"></line>
                            <line x1="4" x2="20" y1="6" y2="6"></line>
                            <line x1="4" x2="20" y1="18" y2="18"></line>
                        </svg>
                    </button>
                </div>
            </div>
            
            <!-- Main Content -->
            <div class="flex-1 relative overflow-auto pt-16 lg:pt-0">
                <div class="min-h-full p-4 md:p-6 lg:p-8 bg-white w-full">
                    <!-- Tab Content -->
                    <div class="w-full">
                        <!-- Tab Navigation -->
                        <div class="flex space-x-1 mb-6">
                            <button onclick="showTab('dashboard')" id="dashboard-tab" class="px-4 py-2 text-sm font-medium rounded-md bg-blue-50 text-blue-700 border border-blue-200">
                                Kontrol Paneli
                            </button>
                            <button onclick="showTab('personal')" id="personal-tab" class="px-4 py-2 text-sm font-medium rounded-md text-gray-600 hover:text-gray-900 hover:bg-gray-100 border border-gray-200">
                                Kişisel Bilgiler
                            </button>
                            <button onclick="showTab('security')" id="security-tab" class="px-4 py-2 text-sm font-medium rounded-md text-gray-600 hover:text-gray-900 hover:bg-gray-100 border border-gray-200">
                                Güvenlik
                            </button>
                            <button onclick="showTab('notifications')" id="notifications-tab" class="px-4 py-2 text-sm font-medium rounded-md text-gray-600 hover:text-gray-900 hover:bg-gray-100 border border-gray-200">
                                Bildirimler
                            </button>
                        </div>

                        <!-- Dashboard Tab -->
                        <div id="dashboard-content" class="tab-content">
                            <div class="space-y-6 w-full">
                                <div class="flex items-center justify-between">
                                    <h2 class="text-3xl font-bold text-gray-900">Kontrol Paneli</h2>
                                </div>
                            
                            <!-- Account Summary -->
                            <div class="rounded-md border border-gray-200 bg-white shadow-sm">
                                <div class="p-6">
                                    <div class="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
                                        <div>
                                            <h3 class="text-xl font-semibold text-gray-900">Hesap Özetiniz</h3>
                                            <p class="text-gray-600 mt-1">
                                                Hesap Durumu: <span class="text-green-600 font-medium"><?php echo strtoupper($user['status']); ?></span> • 
                                                Üyelik başlangıcı: <?php echo date('d.m.Y H:i:s', strtotime($user['created_at'])); ?>
                                            </p>
                                        </div>
                                        <a href="#kyc">
                                            <button class="inline-flex items-center justify-center rounded-md text-sm font-medium bg-blue-600 text-white hover:bg-blue-700 h-10 px-4 py-2 gap-1.5">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4">
                                                    <path d="M20 13c0 5-3.5 7.5-7.66 8.95a1 1 0 0 1-.67-.01C7.5 20.5 4 18 4 13V6a1 1 0 0 1 1-1c2 0 4.5-1.2 6.24-2.72a1.17 1.17 0 0 1 1.52 0C14.51 3.81 17 5 19 5a1 1 0 0 1 1 1z"></path>
                                                </svg>
                                                KYC Doğrulaması
                                            </button>
                                        </a>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Stats Grid -->
                            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                                <!-- Security Score -->
                                <div class="h-full">
                                    <div class="rounded-md border border-gray-200 bg-white shadow-sm h-full">
                                        <div class="p-6 flex flex-col h-full">
                                            <div class="flex justify-between items-start">
                                                <div>
                                                    <p class="text-sm font-medium text-gray-600 mb-1">Güvenlik Skoru</p>
                                                    <h3 class="text-2xl font-bold text-gray-900">30/100</h3>
                                                    <p class="text-sm text-gray-600 mt-1">Orta</p>
                                                </div>
                                                <div class="p-3 rounded-full bg-red-100 text-red-600">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5">
                                                        <path d="M20 13c0 5-3.5 7.5-7.66 8.95a1 1 0 0 1-.67-.01C7.5 20.5 4 18 4 13V6a1 1 0 0 1 1-1c2 0 4.5-1.2 6.24-2.72a1.17 1.17 0 0 1 1.52 0C14.51 3.81 17 5 19 5a1 1 0 0 1 1 1z"></path>
                                                    </svg>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Account Age -->
                                <div class="h-full">
                                    <div class="rounded-md border border-gray-200 bg-white shadow-sm h-full">
                                        <div class="p-6 flex flex-col h-full">
                                            <div class="flex justify-between items-start">
                                                <div>
                                                    <p class="text-sm font-medium text-gray-600 mb-1">Hesap Yaşı</p>
                                                    <h3 class="text-2xl font-bold text-gray-900"><?php echo floor((time() - strtotime($user['created_at'])) / (60 * 60 * 24)); ?> gün</h3>
                                                </div>
                                                <div class="p-3 rounded-full bg-blue-100 text-blue-600">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5">
                                                        <path d="M8 2v4"></path>
                                                        <path d="M16 2v4"></path>
                                                        <rect width="18" height="18" x="3" y="4" rx="2"></rect>
                                                        <path d="M3 10h18"></path>
                                                    </svg>
                                                </div>
                                            </div>
                                            <div class="flex items-center mt-auto pt-2">
                                                <span class="inline-flex items-center justify-center rounded-md border border-green-200 bg-green-50 text-green-700 px-2 py-0.5 text-xs font-medium">+100% Yaşam Boyu</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- API Keys -->
                                <div class="h-full">
                                    <div class="rounded-md border border-gray-200 bg-white shadow-sm h-full">
                                        <div class="p-6 flex flex-col h-full">
                                            <div class="flex justify-between items-start">
                                                <div>
                                                    <p class="text-sm font-medium text-gray-600 mb-1">API Anahtarları</p>
                                                    <h3 class="text-2xl font-bold text-gray-900">0</h3>
                                                    <p class="text-sm text-gray-600 mt-1">Anahtar oluşturulmadı</p>
                                                </div>
                                                <div class="p-3 rounded-full bg-purple-100 text-purple-600">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5">
                                                        <path d="m15.5 7.5 2.3 2.3a1 1 0 0 0 1.4 0l2.1-2.1a1 1 0 0 0 0-1.4L19 4"></path>
                                                        <path d="m21 2-9.6 9.6"></path>
                                                        <circle cx="7.5" cy="15.5" r="5.5"></circle>
                                                    </svg>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Additional Stats -->
                            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                                <!-- Login Activity -->
                                <div class="h-full">
                                    <div class="rounded-md border border-gray-200 bg-white shadow-sm h-full">
                                        <div class="p-6 flex flex-col h-full">
                                            <div class="flex justify-between items-start">
                                                <div>
                                                    <p class="text-sm font-medium text-gray-600 mb-1">Giriş Aktivitesi</p>
                                                    <h3 class="text-2xl font-bold text-gray-900">4</h3>
                                                    <p class="text-sm text-gray-600 mt-1">Bu ay girişler</p>
                                                </div>
                                                <div class="p-3 rounded-full bg-blue-100 text-blue-600">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5">
                                                        <path d="M22 12h-2.48a2 2 0 0 0-1.93 1.46l-2.35 8.36a.25.25 0 0 1-.48 0L9.24 2.18a.25.25 0 0 0-.48 0l-2.35 8.36A2 2 0 0 1 4.49 12H2"></path>
                                                    </svg>
                                                </div>
                                            </div>
                                            <div class="flex items-center mt-auto pt-2">
                                                <span class="inline-flex items-center justify-center rounded-md border border-green-200 bg-green-50 text-green-700 px-2 py-0.5 text-xs font-medium">+33% geçen aya göre</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- KYC Level -->
                                <div class="h-full">
                                    <div class="rounded-md border border-gray-200 bg-white shadow-sm h-full">
                                        <div class="p-6 flex flex-col h-full">
                                            <div class="flex justify-between items-start">
                                                <div>
                                                    <p class="text-sm font-medium text-gray-600 mb-1">KYC Seviyesi</p>
                                                    <h3 class="text-2xl font-bold text-gray-900">0</h3>
                                                    <p class="text-sm text-gray-600 mt-1">Temel doğrulama</p>
                                                </div>
                                                <div class="p-3 rounded-full bg-green-100 text-green-600">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5">
                                                        <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"></path>
                                                        <circle cx="9" cy="7" r="4"></circle>
                                                        <path d="M22 21v-2a4 4 0 0 0-3-3.87"></path>
                                                        <path d="M16 3.13a4 4 0 0 1 0 7.75"></path>
                                                    </svg>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Wallet Status -->
                                <div class="h-full">
                                    <div class="rounded-md border border-gray-200 bg-white shadow-sm h-full">
                                        <div class="p-6 flex flex-col h-full">
                                            <div class="flex justify-between items-start">
                                                <div>
                                                    <p class="text-sm font-medium text-gray-600 mb-1">Cüzdan Durumu</p>
                                                    <h3 class="text-2xl font-bold text-gray-900">Bağlı Değil</h3>
                                                    <p class="text-sm text-gray-600 mt-1">Cüzdanınızı bağlayın</p>
                                                </div>
                                                <div class="p-3 rounded-full bg-amber-100 text-amber-600">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5">
                                                        <path d="M19 7V4a1 1 0 0 0-1-1H5a2 2 0 0 0 0 4h15a1 1 0 0 1 1 1v4h-3a2 2 0 0 0 0 4h3a1 1 0 0 0 1-1v-2a1 1 0 0 0-1-1"></path>
                                                        <path d="M3 5v14a2 2 0 0 0 2 2h15a1 1 0 0 0 1-1v-4"></path>
                                                    </svg>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Activity and Security Overview -->
                            <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                                <!-- Recent Activity -->
                                <div class="rounded-md border border-gray-200 bg-white shadow-sm h-full overflow-hidden">
                                    <div class="flex flex-col space-y-1.5 p-6 pb-2">
                                        <div class="flex justify-between items-center">
                                            <div class="font-semibold leading-none tracking-tight text-gray-900">Son Aktiviteler</div>
                                            <button class="inline-flex items-center justify-center gap-2 text-sm font-medium hover:bg-gray-100 rounded-md h-8 px-2 text-gray-600 hover:text-gray-900">
                                                Tümünü Gör
                                                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 ml-1">
                                                    <path d="m9 18 6-6-6-6"></path>
                                                </svg>
                                            </button>
                                        </div>
                                    </div>
                                    <div class="p-6 pt-0">
                                        <div class="space-y-1">
                                            <div class="text-center py-8 text-gray-600">Görüntülenecek yeni etkinlik yok</div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Security Overview -->
                                <div class="rounded-md border border-gray-200 bg-white shadow-sm h-full overflow-hidden">
                                    <div class="flex flex-col space-y-1.5 p-6 pb-2">
                                        <div class="flex justify-between items-center">
                                            <div class="font-semibold leading-none tracking-tight text-gray-900">Güvenliğe Genel Bakış</div>
                                        </div>
                                    </div>
                                    <div class="p-6 pt-0">
                                        <div class="space-y-6">
                                            <div class="flex items-center justify-between">
                                                <div>
                                                    <h3 class="font-medium text-gray-900">Güvenlik Puanı</h3>
                                                    <p class="text-2xl font-bold text-red-600">Orta</p>
                                                </div>
                                                <div class="relative h-20 w-20 cursor-help">
                                                    <svg class="h-full w-full -rotate-90" viewBox="0 0 36 36">
                                                        <circle cx="18" cy="18" r="16" fill="none" class="stroke-gray-300" stroke-width="3"></circle>
                                                        <circle cx="18" cy="18" r="16" fill="none" class="stroke-red-500" stroke-width="3" stroke-dasharray="100" stroke-dashoffset="70"></circle>
                                                    </svg>
                                                    <div class="absolute inset-0 flex items-center justify-center text-lg font-medium text-gray-900">30</div>
                                                </div>
                                            </div>
                                            <div class="space-y-3">
                                                <div class="flex items-center justify-between p-2 rounded-lg hover:bg-gray-50 transition-colors duration-200">
                                                    <div class="flex items-center">
                                                        <div class="p-1.5 rounded-full bg-gray-100 mr-3 shadow-sm">
                                                            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 text-gray-500">
                                                                <path d="M18 6 6 18"></path>
                                                                <path d="m6 6 12 12"></path>
                                                            </svg>
                                                        </div>
                                                        <span class="text-sm font-medium text-gray-900">İki faktörlü kimlik doğrulama</span>
                                                    </div>
                                                    <button class="inline-flex items-center justify-center gap-2 font-medium border border-gray-300 bg-white hover:bg-gray-50 hover:text-gray-900 rounded-md h-8 px-3 text-xs shadow-sm">Etkinleştirmek</button>
                                                </div>
                                                <div class="flex items-center justify-between p-2 rounded-lg hover:bg-gray-50 transition-colors duration-200">
                                                    <div class="flex items-center">
                                                        <div class="p-1.5 rounded-full bg-gray-100 mr-3 shadow-sm">
                                                            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 text-gray-500">
                                                                <path d="M18 6 6 18"></path>
                                                                <path d="m6 6 12 12"></path>
                                                            </svg>
                                                        </div>
                                                        <span class="text-sm font-medium text-gray-900">E-posta Doğrulandı</span>
                                                    </div>
                                                    <button class="inline-flex items-center justify-center gap-2 font-medium border border-gray-300 bg-white hover:bg-gray-50 hover:text-gray-900 rounded-md h-8 px-3 text-xs shadow-sm">Doğrulamak</button>
                                                </div>
                                                <div class="flex items-center justify-between p-2 rounded-lg hover:bg-gray-50 transition-colors duration-200">
                                                    <div class="flex items-center">
                                                        <div class="p-1.5 rounded-full bg-gray-100 mr-3 shadow-sm">
                                                            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 text-gray-500">
                                                                <path d="M18 6 6 18"></path>
                                                                <path d="m6 6 12 12"></path>
                                                            </svg>
                                                        </div>
                                                        <span class="text-sm font-medium text-gray-900">Telefon Doğrulandı</span>
                                                    </div>
                                                    <button class="inline-flex items-center justify-center gap-2 font-medium border border-gray-300 bg-white hover:bg-gray-50 hover:text-gray-900 rounded-md h-8 px-3 text-xs shadow-sm">Doğrulamak</button>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        </div>

                        <!-- Personal Info Tab -->
                        <div id="personal-content" class="tab-content">
                            <div class="space-y-6 w-full">
                                <div class="flex items-center justify-between">
                                    <div class="space-y-1">
                                        <h2 class="text-2xl font-bold text-gray-900">Kişisel Bilgiler</h2>
                                        <p class="text-gray-600">Kişisel bilgilerinizi ve hesap tercihlerinizi yönetin</p>
                                    </div>
                                </div>

                                <!-- Profile Details -->
                                <div class="rounded-md border border-gray-200 bg-white shadow-sm overflow-hidden">
                                    <div class="flex flex-col space-y-1.5 p-6 bg-blue-50/50 border-b border-gray-200">
                                        <div class="flex justify-between items-center">
                                            <div>
                                                <div class="font-semibold leading-none tracking-tight text-gray-900">Profil Detayları</div>
                                                <div class="text-sm text-gray-600">Kişisel bilgilerinizi ve iletişim bilgilerinizi görüntüleyin</div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="p-6">
                                        <?php if (!empty($update_success)): ?>
                                            <div class="mb-4 px-4 py-2 rounded-md text-sm bg-green-50 text-green-700 border border-green-200">Bilgileriniz güncellendi.</div>
                                        <?php endif; ?>
                                        <?php if (!empty($update_error)): ?>
                                            <div class="mb-4 px-4 py-2 rounded-md text-sm bg-red-50 text-red-700 border border-red-200"><?php echo htmlspecialchars($update_error); ?></div>
                                        <?php endif; ?>
                                        <form method="POST">
                                            <input type="hidden" name="action" value="update_personal">
                                        <div class="flex flex-col lg:flex-row gap-6 lg:gap-8">
                                            <!-- Profile Picture -->
                                            <div class="w-full lg:w-1/3 flex flex-col items-center">
                                                <div class="relative group">
                                                    <div class="relative cursor-pointer">
                                                        <span class="relative flex w-32 h-32 lg:h-40 lg:w-40 overflow-hidden rounded-full border-4 border-white shadow-md mb-4">
                                                            <?php 
                                                            $defaultAvatarProfile = "data:image/svg+xml;utf8,<svg xmlns='http://www.w3.org/2000/svg' width='80' height='80'><rect width='100%' height='100%' fill='%23e5e7eb'/><text x='50%' y='52%' dominant-baseline='middle' text-anchor='middle' fill='%239ca3af' font-family='Arial' font-size='28'>👤</text></svg>";
                                                            $avatarSrcProfile = !empty($user['profile_image']) ? htmlspecialchars($user['profile_image']) : $defaultAvatarProfile;
                                                            ?>
                                                            <img id="profile-main-avatar" class="w-full h-full object-cover" alt="<?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?>" src="<?php echo $avatarSrcProfile; ?>">
                                                        </span>
                                                        <div class="absolute inset-0 bg-black/50 rounded-full opacity-0 group-hover:opacity-100 transition-opacity duration-200 flex items-center justify-center">
                                                            <div class="flex flex-col items-center justify-center text-white">
                                                                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-8 w-8 mb-1">
                                                                    <path d="M14.5 4h-5L7 7H4a2 2 0 0 0-2 2v9a2 2 0 0 0 2 2h16a2 2 0 0 0 2-2V9a2 2 0 0 0-2-2h-3l-2.5-3z"></path>
                                                                    <circle cx="12" cy="13" r="3"></circle>
                                                                </svg>
                                                                <span class="text-xs font-medium">Fotoğrafı Değiştir</span>
                                                            </div>
                                                        </div>
                                                        <input type="file" id="profileImageInput" accept="image/*" class="absolute inset-0 w-full h-full opacity-0 cursor-pointer" onchange="uploadProfileImage(this)">
                                                    </div>
                                                </div>
                                                <div class="text-center space-y-1 mb-4">
                                                    <h3 class="text-xl font-semibold text-gray-900"><?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?></h3>
                                                    <p class="text-gray-600"><?php echo htmlspecialchars($user['email']); ?></p>
                                                </div>
                                                <div class="flex flex-wrap justify-center gap-2 mb-4">
                                                    <span class="inline-flex items-center justify-center rounded-md border border-blue-200 bg-blue-50 text-blue-700 px-2 py-0.5 text-xs font-medium">
                                                        <svg class="h-3 w-3 mr-1" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M12 22C17.5228 22 22 17.5228 22 12C22 6.47715 17.5228 2 12 2C6.47715 2 2 6.47715 2 12C2 17.5228 6.47715 22 12 22Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M12 16V12" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M12 8H12.01" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                                        </svg>
                                                        Seviye 0
                                                    </span>
                                                </div>
                                                <div class="w-full space-y-3 px-4">
                                                    <div class="flex items-center gap-2 text-sm">
                                                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 text-gray-500">
                                                            <path d="M8 2v4"></path>
                                                            <path d="M16 2v4"></path>
                                                            <rect width="18" height="18" x="3" y="4" rx="2"></rect>
                                                            <path d="M3 10h18"></path>
                                                        </svg>
                                                        <span class="text-gray-600">Üyelik başlangıcı:</span>
                                                        <span class="font-medium text-gray-900"><?php echo date('d.m.Y H:i', strtotime($user['created_at'])); ?></span>
                                                    </div>
                                                    <div class="flex items-center gap-2 text-sm">
                                                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 text-gray-500">
                                                            <path d="M20 10c0 4.993-5.539 10.193-7.399 11.799a1 1 0 0 1-1.202 0C9.539 20.193 4 14.993 4 10a8 8 0 0 1 16 0"></path>
                                                            <circle cx="12" cy="10" r="3"></circle>
                                                        </svg>
                                                        <span class="text-gray-600">Yer:</span>
                                                        <span class="font-medium text-gray-900"><?php echo htmlspecialchars($user['country'] ?? 'Belirtilmemiş'); ?></span>
                                                    </div>
                                                </div>
                                            </div>

                                            <!-- Form Fields -->
                                            <div class="w-full lg:w-2/3 space-y-6">
                                                <div class="space-y-4">
                                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                                        <div class="space-y-2">
                                                            <label class="text-sm font-medium text-gray-900 flex items-center gap-2" for="firstName">
                                                                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 text-gray-500">
                                                                    <path d="M19 21v-2a4 4 0 0 0-4-4H9a4 4 0 0 0-4 4v2"></path>
                                                                    <circle cx="12" cy="7" r="4"></circle>
                                                                </svg>
                                                                Ad
                                                            </label>
                                                            <div class="flex-1 w-full flex flex-col">
                                                                <input id="firstName" name="first_name" type="text" value="<?php echo htmlspecialchars($user['first_name']); ?>" class="h-9 w-full rounded-md border border-gray-300 px-3 py-1 bg-white text-gray-900" required>
                                                            </div>
                                                        </div>
                                                        <div class="space-y-2">
                                                            <label class="text-sm font-medium text-gray-900 flex items-center gap-2" for="lastName">
                                                                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 text-gray-500">
                                                                    <path d="M19 21v-2a4 4 0 0 0-4-4H9a4 4 0 0 0-4 4v2"></path>
                                                                    <circle cx="12" cy="7" r="4"></circle>
                                                                </svg>
                                                                Soyadı
                                                            </label>
                                                            <div class="flex-1 w-full flex flex-col">
                                                                <input id="lastName" name="last_name" type="text" value="<?php echo htmlspecialchars($user['last_name']); ?>" class="h-9 w-full rounded-md border border-gray-300 px-3 py-1 bg-white text-gray-900" required>
                                                            </div>
                                                        </div>
                                                        <div class="space-y-2">
                                                            <label class="text-sm font-medium text-gray-900 flex items-center gap-2" for="email">
                                                                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 text-gray-500">
                                                                    <rect width="20" height="16" x="2" y="4" rx="2"></rect>
                                                                    <path d="m22 7-8.97 5.7a1.94 1.94 0 0 1-2.06 0L2 7"></path>
                                                                </svg>
                                                                E-posta Adresi
                                                            </label>
                                                            <div class="relative">
                                                                <div class="flex-1 w-full flex flex-col">
                                                                    <div class="h-9 w-full rounded-md border border-gray-300 px-3 py-1 bg-gray-50 flex items-center">
                                                                        <span class="flex-1 bg-transparent text-gray-500"><?php echo htmlspecialchars($user['email']); ?></span>
                                                                    </div>
                                                                </div>
                                                            </div>
                                                        </div>
                                                        <div class="space-y-2">
                                                            <label class="text-sm font-medium text-gray-900 flex items-center gap-2" for="phone">
                                                                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 text-gray-500">
                                                                    <path d="M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"></path>
                                                                </svg>
                                                                Telefon numarası
                                                            </label>
                                                            <div class="relative">
                                                                <div class="flex-1 w-full flex flex-col">
                                                                    <input id="phone" name="phone" type="text" value="<?php echo htmlspecialchars($user['phone'] ?? ''); ?>" placeholder="Telefon numarası" class="h-9 w-full rounded-md border border-gray-300 px-3 py-1 bg-white text-gray-900">
                                                                </div>
                                                            </div>
                                                        </div>
                                                        <div class="space-y-2">
                                                            <label class="text-sm font-medium text-gray-900 flex items-center gap-2" for="country">
                                                                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 text-gray-500">
                                                                    <path d="M20 10c0 4.993-5.539 10.193-7.399 11.799a1 1 0 0 1-1.202 0C9.539 20.193 4 14.993 4 10a8 8 0 0 1 16 0"></path>
                                                                    <circle cx="12" cy="10" r="3"></circle>
                                                                </svg>
                                                                Ülke
                                                            </label>
                                                            <div class="relative">
                                                                <div class="flex-1 w-full flex flex-col">
                                                                    <input id="country" name="country" type="text" value="<?php echo htmlspecialchars($user['country'] ?? ''); ?>" placeholder="Ülke" class="h-9 w-full rounded-md border border-gray-300 px-3 py-1 bg-white text-gray-900">
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="mt-6">
                                                <button type="submit" class="inline-flex items-center justify-center rounded-md text-sm font-medium bg-blue-600 text-white hover:bg-blue-700 h-10 px-4 py-2">
                                                    Kaydet
                                                </button>
                                            </div>
                                        </form>
                                        </div>
                                    </div>
                                </div>

                                <!-- Account Information -->
                                <div class="rounded-md border border-gray-200 bg-white shadow-sm overflow-hidden">
                                    <div class="flex flex-col space-y-1.5 p-6 bg-blue-50/50 border-b border-gray-200">
                                        <div class="font-semibold leading-none tracking-tight text-gray-900">Hesap Bilgileri</div>
                                        <div class="text-sm text-gray-600">Hesap ayrıntılarınızı ve durumunuzu görüntüleme</div>
                                    </div>
                                    <div class="p-6">
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                            <div class="space-y-2 p-4 rounded-lg bg-gray-50/50 border border-gray-200">
                                                <p class="text-sm font-medium text-gray-600">Hesap Kimliği</p>
                                                <div class="flex items-center">
                                                    <p class="font-medium font-mono text-sm bg-gray-100 py-1 px-2 rounded"><?php echo $user['id']; ?></p>
                                                    <button class="inline-flex items-center justify-center h-8 w-8 ml-2 hover:bg-gray-100 rounded">
                                                        <svg class="h-4 w-4" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z"></path>
                                                        </svg>
                                                    </button>
                                                </div>
                                            </div>
                                            <div class="space-y-2 p-4 rounded-lg bg-gray-50/50 border border-gray-200">
                                                <p class="text-sm font-medium text-gray-600">Doğrulama Düzeyi</p>
                                                <div class="flex items-center gap-2">
                                                    <span class="inline-flex items-center justify-center rounded-md border border-gray-200 bg-gray-50 text-gray-700 px-2 py-0.5 text-xs font-medium">
                                                        <svg class="h-3 w-3 mr-1" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M12 22C17.5228 22 22 17.5228 22 12C22 6.47715 17.5228 2 12 2C6.47715 2 2 6.47715 2 12C2 17.5228 6.47715 22 12 22Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M12 16V12" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M12 8H12.01" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                                        </svg>
                                                        Seviye 0
                                                    </span>
                                                    <a href="#kyc">
                                                        <button class="inline-flex items-center justify-center text-sm font-medium underline-offset-4 hover:underline rounded-md h-auto p-0 text-gray-600">Yükseltmek</button>
                                                    </a>
                                                </div>
                                            </div>
                                            <div class="space-y-2 p-4 rounded-lg bg-gray-50/50 border border-gray-200">
                                                <p class="text-sm font-medium text-gray-600">Üyelik başlangıcı:</p>
                                                <div class="flex items-center gap-2">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 text-gray-600">
                                                        <path d="M8 2v4"></path>
                                                        <path d="M16 2v4"></path>
                                                        <rect width="18" height="18" x="3" y="4" rx="2"></rect>
                                                        <path d="M3 10h18"></path>
                                                    </svg>
                                                    <p class="font-medium text-gray-900"><?php echo date('d.m.Y H:i', strtotime($user['created_at'])); ?></p>
                                                </div>
                                            </div>
                                            <div class="space-y-2 p-4 rounded-lg bg-gray-50/50 border border-gray-200">
                                                <p class="text-sm font-medium text-gray-600">Son giriş:</p>
                                                <div class="flex items-center gap-2">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 text-gray-600">
                                                        <path d="M8 2v4"></path>
                                                        <path d="M16 2v4"></path>
                                                        <rect width="18" height="18" x="3" y="4" rx="2"></rect>
                                                        <path d="M3 10h18"></path>
                                                    </svg>
                                                    <p class="font-medium text-gray-900"><?php echo date('d.m.Y H:i', strtotime($user['updated_at'])); ?></p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Security Tab -->
                        <div id="security-content" class="tab-content" style="display: none;">
                            <div class="space-y-6 w-full">
                                <div class="flex items-center justify-between">
                                    <div class="space-y-1">
                                        <h2 class="text-2xl font-bold text-gray-900">Güvenlik Kontrol Paneli</h2>
                                        <p class="text-gray-600">Hesabınızın güvenlik ayarlarını yönetme</p>
                                    </div>
                                    <div class="flex items-center gap-3 bg-gray-50 p-3 rounded-lg">
                                        <div class="relative h-16 w-16">
                                            <svg viewBox="0 0 100 100" class="h-full w-full">
                                                <circle cx="50" cy="50" r="45" fill="none" stroke="#e5e7eb" stroke-width="10"></circle>
                                                <circle cx="50" cy="50" r="45" fill="none" stroke="#ef4444" stroke-width="10" stroke-dasharray="283" stroke-dashoffset="198.1" transform="rotate(-90 50 50)"></circle>
                                                <text x="50" y="50" dominant-baseline="middle" text-anchor="middle" font-size="24" font-weight="bold" fill="#ef4444">30</text>
                                                <text x="50" y="65" dominant-baseline="middle" text-anchor="middle" font-size="10" fill="#6b7280">%</text>
                                            </svg>
                                        </div>
                                        <div>
                                            <div class="text-sm font-medium text-gray-500">Güvenlik Puanı</div>
                                            <div class="text-xl font-bold text-red-600">Zayıf</div>
                                            <div class="text-xs text-gray-500">2FA'yı etkinleştirerek güvenliğinizi artırın</div>
                                        </div>
                                    </div>
                                </div>

                                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
                                    <!-- 2FA Card -->
                                    <div class="rounded-md border border-gray-200 bg-white shadow-sm h-[200px] flex flex-col">
                                        <div class="flex flex-col space-y-1.5 p-6 pb-2">
                                            <div class="flex items-center justify-between">
                                                <div class="font-semibold tracking-tight text-base">İki Faktörlü Kimlik Doğrulama</div>
                                                <div class="h-8 w-8 rounded-full flex items-center justify-center bg-red-100">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5 text-red-600">
                                                        <path d="M20 13c0 5-3.5 7.5-7.66 8.95a1 1 0 0 1-.67-.01C7.5 20.5 4 18 4 13V6a1 1 0 0 1 1-1c2 0 4.5-1.2 6.24-2.72a1.17 1.17 0 0 1 1.52 0C14.51 3.81 17 5 19 5a1 1 0 0 1 1 1z"></path>
                                                        <path d="M12 8v4"></path>
                                                        <path d="M12 16h.01"></path>
                                                    </svg>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="p-6 pt-0 flex-1 flex flex-col">
                                            <div class="text-sm text-gray-500 mb-4 flex-grow">Ek güvenlik için 2FA'yı etkinleştirin</div>
                                            <div class="flex flex-col gap-2">
                                                <button class="inline-flex items-center justify-center gap-2 text-sm font-medium h-9 rounded-md px-3 w-full bg-green-600 hover:bg-green-700 text-white">
                                                    2FA'yı Etkinleştir
                                                </button>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Password Reset Card -->
                                    <div class="rounded-md border border-gray-200 bg-white shadow-sm h-[200px] flex flex-col">
                                        <div class="flex flex-col space-y-1.5 p-6 pb-2">
                                            <div class="flex items-center justify-between">
                                                <div class="font-semibold tracking-tight text-base">Şifre Sıfırlama</div>
                                                <div class="h-8 w-8 rounded-full bg-blue-100 flex items-center justify-center">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5 text-blue-600">
                                                        <rect width="18" height="11" x="3" y="11" rx="2" ry="2"></rect>
                                                        <path d="M7 11V7a5 5 0 0 1 10 0v4"></path>
                                                    </svg>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="p-6 pt-0 flex-1 flex flex-col">
                                            <div class="text-sm text-gray-500 mb-4 flex-grow">Güvenlik nedeniyle, parola sıfırlama işlemleri e-posta doğrulaması gerektirir</div>
                                            <button class="inline-flex items-center justify-center gap-2 text-sm font-medium border border-gray-300 bg-white hover:bg-gray-50 h-9 rounded-md px-3 w-full">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 mr-2">
                                                    <path d="M9 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h4"></path>
                                                    <polyline points="16 17 21 12 16 7"></polyline>
                                                    <line x1="21" x2="9" y1="12" y2="12"></line>
                                                </svg>
                                                Oturumu Kapat ve Şifreyi Sıfırla
                                            </button>
                                        </div>
                                    </div>
                                </div>

                                <!-- Security Recommendations -->
                                <div class="bg-gray-50 rounded-lg p-4 mb-6">
                                    <h3 class="font-medium mb-2 flex items-center">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5 mr-2 text-blue-600">
                                            <path d="M20 13c0 5-3.5 7.5-7.66 8.95a1 1 0 0 1-.67-.01C7.5 20.5 4 18 4 13V6a1 1 0 0 1 1-1c2 0 4.5-1.2 6.24-2.72a1.17 1.17 0 0 1 1.52 0C14.51 3.81 17 5 19 5a1 1 0 0 1 1 1z"></path>
                                        </svg>
                                        Güvenlik Önerileri
                                    </h3>
                                    <div class="space-y-2">
                                        <div class="flex items-start gap-2">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5 text-amber-500 mt-0.5">
                                                <path d="m21.73 18-8-14a2 2 0 0 0-3.48 0l-8 14A2 2 0 0 0 4 21h16a2 2 0 0 0 1.73-3"></path>
                                                <path d="M12 9v4"></path>
                                                <path d="M12 17h.01"></path>
                                            </svg>
                                            <div class="text-sm">
                                                <span class="font-medium">İki faktörlü kimlik doğrulamayı etkinleştirin</span>
                                                <p class="text-gray-500">Hesabınıza ekstra bir güvenlik katmanı ekleyin</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Password Reset Info -->
                                <div class="relative w-full rounded-lg border px-4 py-3 text-sm bg-blue-50 border-blue-200">
                                    <div class="flex items-start gap-3">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-4 w-4 text-blue-600 mt-0.5">
                                            <circle cx="12" cy="12" r="10"></circle>
                                            <path d="M12 16v-4"></path>
                                            <path d="M12 8h.01"></path>
                                        </svg>
                                        <div>
                                            <div class="font-medium text-blue-800 mb-2">Şifre Sıfırlama İşlemimiz Hakkında</div>
                                            <div class="text-blue-700">
                                                <p class="mb-2">Maksimum güvenlik için, e-posta doğrulaması gerektiren güvenli bir parola sıfırlama akışı uyguladık:</p>
                                                <ol class="list-decimal pl-5 space-y-1 text-sm">
                                                    <li>Hesabınızdan çıkış yapın</li>
                                                    <li>adresindeki giriş sayfasını ziyaret edin <span class="font-medium">/login</span></li>
                                                    <li>"Şifremi Unuttum" bağlantısına tıklayın</li>
                                                    <li>Sıfırlama bağlantısı almak için e-posta adresinizi girin</li>
                                                    <li>E-postanızı kontrol edin ve güvenli sıfırlama bağlantısını tıklayın</li>
                                                    <li>Yeni bir şifre oluşturun ve giriş yapın</li>
                                                </ol>
                                                <p class="mt-2 text-sm">Bu yaklaşım, yetkisiz parola değişikliklerini önler ve en iyi güvenlik uygulamalarını izler.</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Notifications Tab -->
                        <div id="notifications-content" class="tab-content" style="display: none;">
                            <div class="space-y-6 w-full">
                                <div class="flex items-center justify-between">
                                    <div class="space-y-1">
                                        <h2 class="text-2xl font-bold text-gray-900">Bildirim Tercihleri</h2>
                                        <p class="text-gray-600">Bildirim ayarlarınızı yönetin</p>
                                    </div>
                                </div>

                                <!-- Notification Settings -->
                                <div class="rounded-md border border-gray-200 bg-white shadow-sm overflow-hidden">
                                    <div class="flex flex-col space-y-1.5 p-6 bg-blue-50/50 border-b border-gray-200">
                                        <div class="font-semibold leading-none tracking-tight text-gray-900">İletişim Kanalları</div>
                                        <div class="text-sm text-gray-600">Bildirimleri nasıl almak istediğinizi seçin</div>
                                    </div>
                                    <div class="p-6 pt-0 space-y-6">
                                        <div class="space-y-4">
                                            <!-- Email Notifications -->
                                            <div class="flex items-center justify-between p-3 rounded-lg border border-gray-200 hover:bg-gray-50 transition-colors">
                                                <div class="flex items-start gap-3">
                                                    <div class="bg-blue-100 p-2 rounded-full">
                                                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5 text-blue-600">
                                                            <rect width="20" height="16" x="2" y="4" rx="2"></rect>
                                                            <path d="m22 7-8.97 5.7a1.94 1.94 0 0 1-2.06 0L2 7"></path>
                                                        </svg>
                                                    </div>
                                                    <div>
                                                        <div class="font-medium text-gray-900">E-posta Bildirimleri</div>
                                                        <div class="text-sm text-gray-500">E-posta yoluyla bildirimleri şu adresten alın: <?php echo htmlspecialchars($user['email']); ?></div>
                                                    </div>
                                                </div>
                                                <label class="relative inline-flex items-center cursor-pointer">
                                                    <input type="checkbox" class="sr-only peer" checked>
                                                    <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                                                </label>
                                            </div>

                                            <!-- SMS Notifications -->
                                            <div class="flex items-center justify-between p-3 rounded-lg border border-gray-200 hover:bg-gray-50 transition-colors">
                                                <div class="flex items-start gap-3">
                                                    <div class="bg-green-100 p-2 rounded-full">
                                                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5 text-green-600">
                                                            <path d="M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"></path>
                                                        </svg>
                                                    </div>
                                                    <div>
                                                        <div class="font-medium text-gray-900">SMS Bildirimleri</div>
                                                        <div class="text-sm text-gray-500">SMS ile bildirim almak için telefon numarası ayarlanmamış</div>
                                                    </div>
                                                </div>
                                                <label class="relative inline-flex items-center cursor-pointer">
                                                    <input type="checkbox" class="sr-only peer">
                                                    <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                                                </label>
                                            </div>

                                            <!-- Push Notifications -->
                                            <div class="flex items-center justify-between p-3 rounded-lg border border-gray-200 hover:bg-gray-50 transition-colors">
                                                <div class="flex items-start gap-3">
                                                    <div class="bg-purple-100 p-2 rounded-full">
                                                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="h-5 w-5 text-purple-600">
                                                            <path d="M10.268 21a2 2 0 0 0 3.464 0"></path>
                                                            <path d="M3.262 15.326A1 1 0 0 0 4 17h16a1 1 0 0 0 .74-1.673C19.41 13.956 18 12.499 18 8A6 6 0 0 0 6 8c0 4.499-1.411 5.956-2.738 7.326"></path>
                                                        </svg>
                                                    </div>
                                                    <div>
                                                        <div class="font-medium text-gray-900">Anlık Bildirimler</div>
                                                        <div class="text-sm text-gray-500">Cihazlarınızda bildirim alma</div>
                                                    </div>
                                                </div>
                                                <label class="relative inline-flex items-center cursor-pointer">
                                                    <input type="checkbox" class="sr-only peer" checked>
                                                    <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                                                </label>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Notification Types -->
                                <div class="rounded-md border border-gray-200 bg-white shadow-sm overflow-hidden">
                                    <div class="flex flex-col space-y-1.5 p-6 bg-blue-50/50 border-b border-gray-200">
                                        <div class="font-semibold leading-none tracking-tight text-gray-900">Bildirim Türleri</div>
                                        <div class="text-sm text-gray-600">Hangi konularda bildirim almak istediğinizi seçin</div>
                                    </div>
                                    <div class="p-6 pt-0 space-y-4">
                                        <div class="space-y-3">
                                            <div class="flex items-center justify-between">
                                                <div>
                                                    <div class="font-medium text-gray-900">İşlem Bildirimleri</div>
                                                    <div class="text-sm text-gray-500">Alım/satım işlemleri hakkında bildirimler</div>
                                                </div>
                                                <label class="relative inline-flex items-center cursor-pointer">
                                                    <input type="checkbox" class="sr-only peer" checked>
                                                    <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                                                </label>
                                            </div>
                                            <div class="flex items-center justify-between">
                                                <div>
                                                    <div class="font-medium text-gray-900">Güvenlik Bildirimleri</div>
                                                    <div class="text-sm text-gray-500">Hesap güvenliği ile ilgili bildirimler</div>
                                                </div>
                                                <label class="relative inline-flex items-center cursor-pointer">
                                                    <input type="checkbox" class="sr-only peer" checked>
                                                    <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                                                </label>
                                            </div>
                                            <div class="flex items-center justify-between">
                                                <div>
                                                    <div class="font-medium text-gray-900">Piyasa Bildirimleri</div>
                                                    <div class="text-sm text-gray-500">Kripto para fiyat hareketleri hakkında bildirimler</div>
                                                </div>
                                                <label class="relative inline-flex items-center cursor-pointer">
                                                    <input type="checkbox" class="sr-only peer">
                                                    <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                                                </label>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <?php include 'components/footer.php'; ?>

    <script>
        function showTab(tabName) {
            // Hide all tab contents
            document.getElementById('dashboard-content').style.display = 'none';
            document.getElementById('personal-content').style.display = 'none';
            document.getElementById('security-content').style.display = 'none';
            document.getElementById('notifications-content').style.display = 'none';

            // Remove active class from all top tabs
            document.getElementById('dashboard-tab').className = 'px-4 py-2 text-sm font-medium rounded-md text-gray-600 hover:text-gray-900 hover:bg-gray-100 border border-gray-200';
            document.getElementById('personal-tab').className = 'px-4 py-2 text-sm font-medium rounded-md text-gray-600 hover:text-gray-900 hover:bg-gray-100 border border-gray-200';
            document.getElementById('security-tab').className = 'px-4 py-2 text-sm font-medium rounded-md text-gray-600 hover:text-gray-900 hover:bg-gray-100 border border-gray-200';
            document.getElementById('notifications-tab').className = 'px-4 py-2 text-sm font-medium rounded-md text-gray-600 hover:text-gray-900 hover:bg-gray-100 border border-gray-200';

            // Remove active class from all sidebar links
            const sidebarLinks = document.querySelectorAll('nav a');
            sidebarLinks.forEach(link => {
                if (link.onclick && link.onclick.toString().includes('showTab')) {
                    link.className = 'flex items-center gap-3 px-3 py-2.5 rounded-lg text-sm font-medium text-gray-600 hover:text-gray-900 hover:bg-gray-100 cursor-pointer';
                    const iconDiv = link.querySelector('div');
                    if (iconDiv) {
                        iconDiv.className = 'p-1.5 rounded-md bg-gray-100';
                    }
                }
            });

            // Show selected tab content
            document.getElementById(tabName + '-content').style.display = 'block';

            // Add active class to selected top tab
            document.getElementById(tabName + '-tab').className = 'px-4 py-2 text-sm font-medium rounded-md bg-blue-50 text-blue-700 border border-blue-200';

            // Add active class to selected sidebar link
            const activeSidebarLink = document.querySelector(`nav a[onclick="showTab('${tabName}')"]`);
            if (activeSidebarLink) {
                activeSidebarLink.className = 'flex items-center gap-3 px-3 py-2.5 rounded-lg text-sm font-medium bg-blue-50 text-blue-700 cursor-pointer';
                const iconDiv = activeSidebarLink.querySelector('div');
                if (iconDiv) {
                    iconDiv.className = 'p-1.5 rounded-md bg-blue-100';
                }
            }
        }


        // Profile image upload
        function uploadProfileImage(input) {
            const file = input.files[0];
            if (!file) return;

            // Validate file type
            if (!file.type.startsWith('image/')) {
                showMessage('Lütfen sadece resim dosyası seçin.', 'error');
                return;
            }

            // Validate file size (max 5MB)
            if (file.size > 5 * 1024 * 1024) {
                showMessage('Dosya boyutu 5MB\'dan küçük olmalıdır.', 'error');
                return;
            }

            // Show loading
            showMessage('Profil fotoğrafı yükleniyor...', 'info');

            // Upload to server
            const formData = new FormData();
            formData.append('profile_image', file);

            fetch('upload_profile_image.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const cacheBuster = '?t=' + Date.now();
                    
                    // Update main profile image
                    const profileMainAvatar = document.getElementById('profile-main-avatar');
                    if (profileMainAvatar) profileMainAvatar.src = data.image_path + cacheBuster;
                    
                    // Update header avatars
                    const headerAvatar = document.getElementById('user-avatar');
                    const dropdownAvatar = document.getElementById('dropdown-avatar');
                    const sidebarAvatar = document.getElementById('sidebar-avatar');
                    
                    if (headerAvatar) headerAvatar.src = data.image_path + cacheBuster;
                    if (dropdownAvatar) dropdownAvatar.src = data.image_path + cacheBuster;
                    if (sidebarAvatar) sidebarAvatar.src = data.image_path + cacheBuster;
                    
                    showMessage(data.message, 'success');
                } else {
                    showMessage(data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Upload error:', error);
                showMessage('Dosya yüklenirken hata oluştu.', 'error');
            });
        }

        // Show message function
        function showMessage(message, type = 'info') {
            const messageDiv = document.createElement('div');
            messageDiv.className = `fixed top-4 right-4 z-50 px-4 py-2 rounded-md text-white ${
                type === 'success' ? 'bg-green-500' : 
                type === 'error' ? 'bg-red-500' : 'bg-blue-500'
            }`;
            messageDiv.textContent = message;
            
            document.body.appendChild(messageDiv);
            
            setTimeout(() => {
                messageDiv.remove();
            }, 3000);
        }

        // Initialize with dashboard tab active
        document.addEventListener('DOMContentLoaded', function() {
            showTab('dashboard');
        });
    </script>
</body>
</html>
