<?php 
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}
require_once 'config/database.php';
$user_id = $_SESSION['user_id'];

// Kullanıcı lending pozisyonları
try {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM lendings WHERE user_id = ? ORDER BY created_at DESC");
    $stmt->execute([$user_id]);
    $user_lendings = $stmt->fetchAll();
} catch (Exception $e) {
    $user_lendings = [];
}

include 'components/header.php'; ?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Borç Verme - KriptoCum</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/style.css">
    <script src="assets/js/dropdown.js"></script>
</head>
<body class="bg-gray-50">
    <!-- Main Content -->
    <div class="pt-16 min-h-screen">
        <div class="container mx-auto px-4 py-8">
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-3xl font-bold text-gray-800 mb-2">Borç Verme</h1>
                <p class="text-gray-600">Kripto para birimlerinizi ödünç verin ve faiz kazanın</p>
            </div>

            <!-- Lending Overview -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Toplam Borç Verilen</p>
                            <p class="text-2xl font-bold text-gray-900" id="totalLent">$0.00</p>
                        </div>
                        <div class="p-3 bg-blue-100 rounded-full">
                            <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Toplam Faiz</p>
                            <p class="text-2xl font-bold text-green-600" id="totalInterest">$0.00</p>
                        </div>
                        <div class="p-3 bg-green-100 rounded-full">
                            <svg class="w-6 h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Aktif Krediler</p>
                            <p class="text-2xl font-bold text-purple-600" id="activeLoans">0</p>
                        </div>
                        <div class="p-3 bg-purple-100 rounded-full">
                            <svg class="w-6 h-6 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Ort. Oran</p>
                            <p class="text-2xl font-bold text-orange-600" id="averageRate">0.0%</p>
                        </div>
                        <div class="p-3 bg-orange-100 rounded-full">
                            <svg class="w-6 h-6 text-orange-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6"></path>
                            </svg>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Lending Markets -->
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden mb-8">
                <div class="p-6 border-b border-gray-200">
                    <h2 class="text-xl font-bold text-gray-800">Borç Verme Piyasaları</h2>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Varlık</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Arz Oranı</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Borç Oranı</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Toplam Arz</th>
                                <th class="px-6 py-3 text-center text-xs font-semibold text-gray-600 uppercase">İşlem</th>
                            </tr>
                        </thead>
                        <tbody id="lendingMarketsTableBody" class="bg-white divide-y divide-gray-200">
                            <!-- Lending markets will be loaded here -->
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- My Lending Positions -->
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                <div class="p-6 border-b border-gray-200">
                    <h2 class="text-xl font-bold text-gray-800">Borç Verme Pozisyonlarım</h2>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Asset</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Lent Amount</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Interest Rate</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Interest Earned</th>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Status</th>
                                <th class="px-6 py-3 text-center text-xs font-semibold text-gray-600 uppercase">Action</th>
                            </tr>
                        </thead>
                        <tbody id="myLendingTableBody" class="bg-white divide-y divide-gray-200">
                            <!-- My lending positions will be loaded here -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Lending Modal -->
    <div id="lendingModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg max-w-md w-full p-6">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-semibold text-gray-800">Lend <span id="lendAssetName">BTC</span></h3>
                    <button onclick="hideLendingModal()" class="text-gray-400 hover:text-gray-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Amount to Lend</label>
                        <input type="number" id="lendAmount" placeholder="0.00" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Interest Rate</label>
                        <select id="interestRate" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="fixed">Fixed Rate (3.5%)</option>
                            <option value="variable">Variable Rate (2.8%)</option>
                        </select>
                    </div>
                    <div class="bg-blue-50 border border-blue-200 rounded-lg p-3">
                        <p class="text-sm text-blue-800">Expected Daily Interest: <span id="expectedInterest">$0.00</span></p>
                        <p class="text-sm text-blue-800">Annual Interest: <span id="annualInterest">$0.00</span></p>
                    </div>
                    <button onclick="confirmLend()" class="w-full bg-blue-600 text-white py-2 rounded-lg hover:bg-blue-700">Lend Now</button>
                </div>
            </div>
        </div>
    </div>

    <script>
        let lendingMarkets = [];
        let myLendingPositions = [];
        let cryptoPrices = {};
        const userLendingsFromPHP = <?php echo json_encode($user_lendings); ?>;

        async function loadLendingData() {
            try {
                const response = await fetch('assets/crypto_prices.json?t=' + Date.now());
                const data = await response.json();
                cryptoPrices = data.prices || {};

                lendingMarkets = Object.entries(cryptoPrices).map(([symbol, priceData]) => ({
                    asset: symbol,
                    name: priceData.name || symbol,
                    supplyRate: Math.random() * 5 + 1,
                    borrowRate: Math.random() * 8 + 3,
                    totalSupply: `${(Math.random() * 10000 + 1000).toFixed(1)} ${symbol}`,
                    logo: priceData.image || null
                }));
            } catch (e) {
                console.error(e);
                lendingMarkets = [];
            }

            // PHP verisini JS modeline dönüştür
            myLendingPositions = userLendingsFromPHP.map(l => ({
                id: l.id,
                asset: l.coin_symbol,
                name: l.coin_symbol,
                lentAmount: parseFloat(l.amount),
                interestRate: parseFloat(l.interest_rate),
                interestEarned: parseFloat(l.interest_earned),
                status: l.status === 'active' ? 'Active' : 'Closed',
                logo: (cryptoPrices[l.coin_symbol] && cryptoPrices[l.coin_symbol].image) ? cryptoPrices[l.coin_symbol].image : null
            }));

            updateLendingMarketsTable();
            updateMyLendingTable();
            updateOverview();
        }

        function updateLendingMarketsTable() {
            const tbody = document.getElementById('lendingMarketsTableBody');
            tbody.innerHTML = '';
            lendingMarkets.forEach(market => {
                const tr = document.createElement('tr');
                tr.className = 'hover:bg-gray-50';
                tr.innerHTML = `
                    <td class="px-6 py-4">
                        <div class="flex items-center gap-3">
                            <div class="w-8 h-8 rounded-full overflow-hidden bg-gray-100 flex items-center justify-center">
                                ${market.logo ? `<img src="${market.logo}" alt="${market.asset}" class="w-6 h-6 object-contain">` : `<div class="w-6 h-6 rounded-full bg-gray-200 text-gray-700 text-[10px] font-bold flex items-center justify-center">${market.asset.substring(0,2)}</div>`}
                            </div>
                            <div>
                                <div class="font-semibold text-gray-800">${market.asset}</div>
                                <div class="text-xs text-gray-500">${market.name}</div>
                            </div>
                        </div>
                    </td>
                    <td class="px-6 py-4 text-right text-green-600 font-semibold">${parseFloat(market.supplyRate).toFixed(1)}%</td>
                    <td class="px-6 py-4 text-right text-red-600 font-semibold">${parseFloat(market.borrowRate).toFixed(1)}%</td>
                    <td class="px-6 py-4 text-right text-gray-800">${market.totalSupply}</td>
                    <td class="px-6 py-4 text-center">
                        <button onclick="showLendingModal('${market.asset}', '${market.name}', ${market.supplyRate})" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 text-sm">Lend</button>
                    </td>
                `;
                tbody.appendChild(tr);
            });
        }

        function updateMyLendingTable() {
            const tbody = document.getElementById('myLendingTableBody');
            tbody.innerHTML = '';
            if (myLendingPositions.length === 0) {
                tbody.innerHTML = '<tr><td colspan="6" class="px-6 py-8 text-center text-gray-500">No active lending positions</td></tr>';
                return;
            }
            myLendingPositions.forEach(position => {
                const tr = document.createElement('tr');
                tr.className = 'hover:bg-gray-50';
                tr.innerHTML = `
                    <td class="px-6 py-4">
                        <div class="flex items-center gap-3">
                            <div class="w-8 h-8 rounded-full overflow-hidden bg-gray-100 flex items-center justify-center">
                                ${position.logo ? `<img src="${position.logo}" alt="${position.asset}" class="w-6 h-6 object-contain">` : `<div class="w-6 h-6 rounded-full bg-gray-200 text-gray-700 text-[10px] font-bold flex items-center justify-center">${position.asset.substring(0,2)}</div>`}
                            </div>
                            <div>
                                <div class="font-semibold text-gray-800">${position.asset}</div>
                                <div class="text-xs text-gray-500">${position.name}</div>
                            </div>
                        </div>
                    </td>
                    <td class="px-6 py-4 text-right text-gray-800 font-mono">${position.lentAmount} ${position.asset}</td>
                    <td class="px-6 py-4 text-right text-green-600 font-semibold">${parseFloat(position.interestRate).toFixed(1)}%</td>
                    <td class="px-6 py-4 text-right text-green-600 font-semibold">${position.interestEarned} ${position.asset}</td>
                    <td class="px-6 py-4">
                        <span class="px-2 py-1 text-xs font-semibold rounded-full ${position.status === 'Active' ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800'}">
                            ${position.status}
                        </span>
                    </td>
                    <td class="px-6 py-4 text-center">
                        <div class="flex justify-center gap-2">
                            <button onclick="claimInterest(${position.id})" class="text-green-600 hover:text-green-700 text-sm">Claim</button>
                            <button onclick="withdrawLending(${position.id}, '${position.asset}')" class="text-red-600 hover:text-red-700 text-sm">Withdraw</button>
                        </div>
                    </td>
                `;
                tbody.appendChild(tr);
            });
        }

        function updateOverview() {
            let totalLent = 0, totalInterest = 0, activeLoans = 0, totalRate = 0;
            myLendingPositions.forEach(position => {
                const price = cryptoPrices[position.asset] ? parseFloat(cryptoPrices[position.asset].raw_price) : 0;
                totalLent += position.lentAmount * price;
                totalInterest += position.interestEarned * price;
                totalRate += position.interestRate;
                if (position.status === 'Active') activeLoans++;
            });
            const averageRate = activeLoans > 0 ? totalRate / activeLoans : 0;
            document.getElementById('totalLent').textContent = `$${totalLent.toFixed(2)}`;
            document.getElementById('totalInterest').textContent = `$${totalInterest.toFixed(2)}`;
            document.getElementById('activeLoans').textContent = activeLoans.toString();
            document.getElementById('averageRate').textContent = `${averageRate.toFixed(1)}%`;
        }

        function showLendingModal(asset, name, rate) {
            document.getElementById('lendAssetName').textContent = asset;
            document.getElementById('lendingModal').classList.remove('hidden');
        }
        function hideLendingModal() { document.getElementById('lendingModal').classList.add('hidden'); }

        function confirmLend() {
            const amount = document.getElementById('lendAmount').value;
            const asset = document.getElementById('lendAssetName').textContent;
            const rateOption = document.getElementById('interestRate').value;
            const rate = rateOption === 'fixed' ? 3.5 : 2.8;
            if (!amount || amount <= 0) { alert('Please enter a valid amount'); return; }
            const fd = new FormData();
            fd.append('coin_symbol', asset); fd.append('amount', amount); fd.append('rate', rate);
            fetch('api/process_lend.php', { method: 'POST', body: fd })
                .then(r=>r.json()).then(d=>{ if(d.success){ alert(d.message); hideLendingModal(); location.reload(); } else { alert('Hata: '+d.message); }})
                .catch(()=>alert('Bir hata oluştu!'));
        }
        function claimInterest(lendingId) {
            const fd = new FormData(); fd.append('lending_id', lendingId);
            fetch('api/process_claim_interest.php', { method: 'POST', body: fd })
                .then(r=>r.json()).then(d=>{ if(d.success){ alert(d.message); location.reload(); } else { alert('Hata: '+d.message); }})
                .catch(()=>alert('Bir hata oluştu!'));
        }
        function withdrawLending(lendingId){
            if(!confirm('Pozisyonu geri çekmek istiyor musunuz?')) return;
            const fd = new FormData(); fd.append('lending_id', lendingId);
            fetch('api/process_withdraw_lending.php', { method: 'POST', body: fd })
                .then(r=>r.json()).then(d=>{ if(d.success){ alert(d.message); location.reload(); } else { alert('Hata: '+d.message); }})
                .catch(()=>alert('Bir hata oluştu!'));
        }

        document.addEventListener('DOMContentLoaded', loadLendingData);
    </script>

    <?php include 'components/footer.php'; ?>
</body>
</html>
