<?php
session_start();
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

require_once '../config/database.php';

try {
    $db = getDB();
    
    // Çekim taleplerini al
    $stmt = $db->prepare("
        SELECT w.*, u.email, u.first_name, u.last_name, ba.bank_name, ba.account_name 
        FROM withdrawals w 
        LEFT JOIN users u ON w.user_id = u.id 
        LEFT JOIN bank_accounts ba ON w.bank_account_id = ba.id 
        ORDER BY w.created_at DESC
    ");
    $stmt->execute();
    $withdrawals = $stmt->fetchAll();
    
} catch (Exception $e) {
    $withdrawals = [];
}

// Çekim onaylama/reddetme
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $withdrawal_id = intval($_POST['withdrawal_id'] ?? 0);
    $action = $_POST['action'] ?? '';
    $admin_notes = $_POST['admin_notes'] ?? '';
    
    if ($withdrawal_id > 0 && in_array($action, ['approve', 'reject'])) {
        try {
            $db->beginTransaction();
            
            // Çekim talebini güncelle
            $new_status = ($action === 'approve') ? 'approved' : 'rejected';
            $stmt = $db->prepare("UPDATE withdrawals SET status = ?, admin_notes = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([$new_status, $admin_notes, $withdrawal_id]);
            
            // Çekim bilgilerini al
            $stmt = $db->prepare("SELECT user_id, coin_symbol, amount FROM withdrawals WHERE id = ?");
            $stmt->execute([$withdrawal_id]);
            $withdrawal = $stmt->fetch();
            
            if ($withdrawal) {
                if ($action === 'approve') {
                    // Onaylandığında frozen'dan çıkar ve balance'dan da düş
                    $stmt = $db->prepare("UPDATE wallets SET frozen = frozen - ?, balance = balance - ? WHERE user_id = ? AND coin_symbol = ? AND frozen >= ?");
                    $result = $stmt->execute([$withdrawal['amount'], $withdrawal['amount'], $withdrawal['user_id'], $withdrawal['coin_symbol'], $withdrawal['amount']]);
                    
                    // Debug log
                    error_log("Withdrawal approve - User: {$withdrawal['user_id']}, Coin: {$withdrawal['coin_symbol']}, Amount: {$withdrawal['amount']}, Result: " . ($result ? 'SUCCESS' : 'FAILED'));
                    
                    // İşlem kaydı ekle
                    $stmt = $db->prepare("INSERT INTO transactions (user_id, type, coin_symbol, amount, price, status, created_at) VALUES (?, 'withdraw', ?, ?, 0, 'completed', NOW())");
                    $stmt->execute([$withdrawal['user_id'], $withdrawal['coin_symbol'], -$withdrawal['amount']]);
                    
                } elseif ($action === 'reject') {
                    // Reddedildiğinde bakiyeyi geri ver (frozen'dan available'a taşı)
                    $stmt = $db->prepare("UPDATE wallets SET frozen = frozen - ?, available = available + ? WHERE user_id = ? AND coin_symbol = ? AND frozen >= ?");
                    $stmt->execute([$withdrawal['amount'], $withdrawal['amount'], $withdrawal['user_id'], $withdrawal['coin_symbol'], $withdrawal['amount']]);
                    
                    // İşlem kaydı ekle
                    $stmt = $db->prepare("INSERT INTO transactions (user_id, type, coin_symbol, amount, price, status, created_at) VALUES (?, 'withdraw_rejected', ?, ?, 0, 'completed', NOW())");
                    $stmt->execute([$withdrawal['user_id'], $withdrawal['coin_symbol'], $withdrawal['amount']]);
                }
            }
            
            $db->commit();
            $success_message = 'Çekim talebi ' . ($action === 'approve' ? 'onaylandı' : 'reddedildi') . '.';
            
        } catch (Exception $e) {
            $db->rollBack();
            $error_message = 'Bir hata oluştu: ' . $e->getMessage();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Çekim Talepleri - Admin Panel</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
</head>
<body class="bg-gray-50">
    <div class="flex h-screen">
        <!-- Sidebar -->
        <div class="w-64 bg-white shadow-lg">
            <div class="p-6">
                <h2 class="text-xl font-bold text-gray-800">Admin Panel</h2>
            </div>
            <nav class="mt-6">
                <a href="index.php" class="block px-6 py-3 text-gray-600 hover:bg-gray-100">Dashboard</a>
                <a href="users.php" class="block px-6 py-3 text-gray-600 hover:bg-gray-100">Kullanıcılar</a>
                <a href="coins.php" class="block px-6 py-3 text-gray-600 hover:bg-gray-100">Coinler</a>
                <a href="bank_accounts.php" class="block px-6 py-3 text-gray-600 hover:bg-gray-100">Banka Hesapları</a>
                <a href="crypto_addresses.php" class="block px-6 py-3 text-gray-600 hover:bg-gray-100">Kripto Adresleri</a>
                <a href="deposits.php" class="block px-6 py-3 text-gray-600 hover:bg-gray-100">Para Yatırma</a>
                <a href="withdrawals.php" class="block px-6 py-3 bg-blue-100 text-blue-600 font-medium">Çekim Talepleri</a>
            </nav>
        </div>

        <!-- Main Content -->
        <div class="flex-1 overflow-y-auto">
            <div class="p-6">
                <!-- Header -->
                <div class="mb-6">
                    <h1 class="text-2xl font-bold text-gray-800">Çekim Talepleri</h1>
                    <p class="text-gray-600">Kullanıcıların çekim taleplerini yönetin</p>
                </div>

                <!-- Success/Error Messages -->
                <?php if (isset($success_message)): ?>
                    <div class="mb-4 p-4 bg-green-100 border border-green-400 text-green-700 rounded-lg">
                        <?php echo htmlspecialchars($success_message); ?>
                    </div>
                <?php endif; ?>

                <?php if (isset($error_message)): ?>
                    <div class="mb-4 p-4 bg-red-100 border border-red-400 text-red-700 rounded-lg">
                        <?php echo htmlspecialchars($error_message); ?>
                    </div>
                <?php endif; ?>

                <!-- Withdrawals Table -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200">
                    <div class="overflow-x-auto">
                        <table class="w-full">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Kullanıcı</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Para Birimi</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Miktar</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Banka</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">IBAN</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Durum</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tarih</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">İşlemler</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php if (empty($withdrawals)): ?>
                                    <tr>
                                        <td colspan="8" class="px-6 py-4 text-center text-gray-500">Henüz çekim talebi bulunmamaktadır.</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($withdrawals as $withdrawal): ?>
                                        <tr class="hover:bg-gray-50">
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <div>
                                                    <div class="text-sm font-medium text-gray-900">
                                                        <?php echo htmlspecialchars($withdrawal['first_name'] . ' ' . $withdrawal['last_name']); ?>
                                                    </div>
                                                    <div class="text-sm text-gray-500"><?php echo htmlspecialchars($withdrawal['email']); ?></div>
                                                </div>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <span class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($withdrawal['coin_symbol']); ?></span>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <?php 
                                                $decimals = ($withdrawal['coin_symbol'] === 'TRY') ? 2 : 8;
                                                $formatted_amount = number_format($withdrawal['amount'], $decimals);
                                                
                                                if ($withdrawal['coin_symbol'] === 'TRY') {
                                                    echo '₺' . $formatted_amount;
                                                } else {
                                                    echo $formatted_amount . ' ' . $withdrawal['coin_symbol'];
                                                }
                                                ?>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <div class="text-sm text-gray-900"><?php echo htmlspecialchars($withdrawal['bank_name'] ?? 'N/A'); ?></div>
                                                <div class="text-sm text-gray-500"><?php echo htmlspecialchars($withdrawal['account_holder']); ?></div>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <div class="text-sm text-gray-900 font-mono"><?php echo htmlspecialchars($withdrawal['iban']); ?></div>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <?php
                                                $status_colors = [
                                                    'pending' => 'bg-yellow-100 text-yellow-800',
                                                    'approved' => 'bg-blue-100 text-blue-800',
                                                    'rejected' => 'bg-red-100 text-red-800',
                                                    'completed' => 'bg-green-100 text-green-800'
                                                ];
                                                $status_texts = [
                                                    'pending' => 'Beklemede',
                                                    'approved' => 'Onaylandı',
                                                    'rejected' => 'Reddedildi',
                                                    'completed' => 'Tamamlandı'
                                                ];
                                                $status = $withdrawal['status'];
                                                ?>
                                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo $status_colors[$status] ?? 'bg-gray-100 text-gray-800'; ?>">
                                                    <?php echo $status_texts[$status] ?? ucfirst($status); ?>
                                                </span>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                                <?php echo date('d.m.Y H:i', strtotime($withdrawal['created_at'])); ?>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                                <?php if ($withdrawal['status'] === 'pending'): ?>
                                                    <div class="flex space-x-2">
                                                        <button onclick="openModal(<?php echo $withdrawal['id']; ?>, 'approve')" class="text-green-600 hover:text-green-900">Onayla</button>
                                                        <button onclick="openModal(<?php echo $withdrawal['id']; ?>, 'reject')" class="text-red-600 hover:text-red-900">Reddet</button>
                                                    </div>
                                                <?php else: ?>
                                                    <span class="text-gray-400">-</span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal -->
    <div id="actionModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg shadow-xl max-w-md w-full">
                <div class="p-6">
                    <h3 class="text-lg font-medium text-gray-900 mb-4" id="modalTitle">İşlem Onayı</h3>
                    <form method="POST" action="">
                        <input type="hidden" name="withdrawal_id" id="modalWithdrawalId">
                        <input type="hidden" name="action" id="modalAction">
                        
                        <div class="mb-4">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Admin Notları</label>
                            <textarea name="admin_notes" id="adminNotes" rows="3" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="İşlem hakkında notlar..."></textarea>
                        </div>
                        
                        <div class="flex justify-end space-x-3">
                            <button type="button" onclick="closeModal()" class="px-4 py-2 text-gray-600 hover:text-gray-800">İptal</button>
                            <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700" id="modalSubmitBtn">Onayla</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <script>
        function openModal(withdrawalId, action) {
            document.getElementById('modalWithdrawalId').value = withdrawalId;
            document.getElementById('modalAction').value = action;
            document.getElementById('modalTitle').textContent = action === 'approve' ? 'Çekim Talebini Onayla' : 'Çekim Talebini Reddet';
            document.getElementById('modalSubmitBtn').textContent = action === 'approve' ? 'Onayla' : 'Reddet';
            document.getElementById('modalSubmitBtn').className = action === 'approve' ? 
                'px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700' : 
                'px-4 py-2 bg-red-600 text-white rounded-md hover:bg-red-700';
            document.getElementById('actionModal').classList.remove('hidden');
        }

        function closeModal() {
            document.getElementById('actionModal').classList.add('hidden');
        }
    </script>
</body>
</html>
