<?php
session_start();

// Admin kontrolü
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$adminEmail = $_SESSION['admin_email'] ?? 'admin';

require_once '../config/database.php';

// Format balance function
function formatBalance($amount, $symbol) {
    if ($amount == 0) return '0';
    
    // TRY için özel format
    if ($symbol === 'TRY') {
        return number_format($amount, 2);
    }
    
    // USDT için özel format (kısaltma yok)
    if ($symbol === 'USDT') {
        return number_format($amount, 2);
    }
    
    // Diğer coinler için akıllı format
    if ($amount >= 1000000) {
        return number_format($amount / 1000000, 2) . 'M';
    } else if ($amount >= 1000) {
        return number_format($amount / 1000, 2) . 'K';
    } else if ($amount >= 1) {
        return number_format($amount, 4);
    } else if ($amount >= 0.01) {
        return number_format($amount, 6);
    } else {
        return number_format($amount, 8);
    }
}

$user_id = (int)($_GET['id'] ?? 0);

if (!$user_id) {
    header('Location: users.php');
    exit;
}

try {
    $db = getDB();
    
    // Kullanıcı bilgilerini çek
    $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch();
    
    if (!$user) {
        header('Location: users.php');
        exit;
    }
    
    // Kullanıcının cüzdanlarını çek
    $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ? ORDER BY balance DESC");
    $stmt->execute([$user_id]);
    $wallets = $stmt->fetchAll();
    
    // Kullanıcının işlemlerini çek
    $stmt = $db->prepare("SELECT * FROM transactions WHERE user_id = ? ORDER BY created_at DESC LIMIT 20");
    $stmt->execute([$user_id]);
    $transactions = $stmt->fetchAll();
    
    // Kripto fiyatlarını yükle
    $crypto_prices = [];
    if (file_exists('../assets/crypto_prices.json')) {
        $price_data = json_decode(file_get_contents('../assets/crypto_prices.json'), true);
        if (isset($price_data['prices'])) {
            $crypto_prices = $price_data['prices'];
        }
    }
    
    // Toplam bakiye hesapla
    $total_balance = 0;
    foreach ($wallets as $wallet) {
        $coin_symbol = $wallet['coin_symbol'];
        $price = 1;
        
        if (isset($crypto_prices[$coin_symbol])) {
            $price = floatval($crypto_prices[$coin_symbol]['raw_price']);
        }
        
        $total_balance += $wallet['balance'] * $price;
    }
    
    // Başarı mesajı varsa göster
    if (isset($_SESSION['admin_flash_success'])) {
        $success = $_SESSION['admin_flash_success'];
        unset($_SESSION['admin_flash_success']);
    }
    
} catch (Exception $e) {
    $user = null;
    $wallets = [];
    $transactions = [];
    $total_balance = 0;
    $error = "Veritabanı hatası: " . $e->getMessage();
}

// POST işlemleri için kripto fiyatlarını yükle (POST işlemlerinde kullanılacak)
$crypto_prices_post = [];
if (file_exists('../assets/crypto_prices.json')) {
    $price_data = json_decode(file_get_contents('../assets/crypto_prices.json'), true);
    if (isset($price_data['prices'])) {
        $crypto_prices_post = $price_data['prices'];
    }
}

// POST işlemleri
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Debug output
    echo "<!-- POST vars: " . var_export($_POST, true) . " -->";
    
    // Kullanıcı durumu değiştirme
    if (isset($_POST['action']) && $_POST['action'] === 'toggle_status') {
        $new_status = (int)($_POST['new_status'] ?? 0);
        
        try {
            $stmt = $db->prepare("UPDATE users SET is_active = ? WHERE id = ?");
            $stmt->execute([$new_status, $user_id]);
            $success = "Kullanıcı durumu güncellendi.";
            // Sayfayı yenile
            header("Location: user_detail.php?id=$user_id");
            exit;
        } catch (Exception $e) {
            $error = "Durum güncellenemedi: " . $e->getMessage();
        }
    }
    
    // Bakiye ekleme (admin)
    if (isset($_POST['action']) && $_POST['action'] === 'add_balance') {
        $coin_symbol = strtoupper(trim($_POST['coin_symbol'] ?? ''));
        $amount = (float)($_POST['amount'] ?? 0);
        
        if ($coin_symbol === '' || $amount <= 0) {
            $error = 'Lütfen geçerli bir coin ve miktar girin.';
        } else {
            try {
            $db->beginTransaction();
            
            $stmt = $db->prepare("SELECT id, balance, available FROM wallets WHERE user_id = ? AND coin_symbol = ? LIMIT 1");
            $stmt->execute([$user_id, $coin_symbol]);
            $existing = $stmt->fetch();
            
            if ($existing) {
                $upd = $db->prepare("UPDATE wallets SET balance = balance + ?, available = available + ? WHERE id = ?");
                $upd->execute([$amount, $amount, $existing['id']]);
                error_log("Wallet updated, ID: " . $existing['id']);
            } else {
                $ins = $db->prepare("INSERT INTO wallets (user_id, coin_symbol, balance, available, frozen) VALUES (?, ?, ?, ?, 0)");
                $ins->execute([$user_id, $coin_symbol, $amount, $amount]);
                error_log("New wallet created for $coin_symbol");
            }
            
            $price = 1.0;
            if (!empty($crypto_prices_post[$coin_symbol]['raw_price'])) {
                $price = (float)$crypto_prices_post[$coin_symbol]['raw_price'];
            } elseif ($coin_symbol === 'USDT') {
                $price = 1.0;
            }
            
            $tx = $db->prepare("INSERT INTO transactions (user_id, type, coin_symbol, amount, price, status, created_at) VALUES (?, 'deposit', ?, ?, ?, 'completed', NOW())");
            $tx->execute([$user_id, $coin_symbol, $amount, $price]);
            
            $db->commit();
            
            error_log("Balance added successfully: $amount $coin_symbol");
            $_SESSION['admin_flash_success'] = 'Bakiye eklendi: ' . $amount . ' ' . $coin_symbol;
            header('Location: user_detail.php?id=' . $user_id);
            exit;
        } catch (Exception $e) {
            if ($db->inTransaction()) { $db->rollBack(); }
            $error = 'Bakiye eklenemedi: ' . $e->getMessage();
            error_log("Balance add error: " . $e->getMessage());
            }
        }
    }
    
    // Bakiye düşürme (admin)
    if (isset($_POST['action']) && $_POST['action'] === 'reduce_balance') {
        $coin_symbol = strtoupper(trim($_POST['coin_symbol'] ?? ''));
        $amount = (float)($_POST['amount'] ?? 0);
        $wallet_id = (int)($_POST['wallet_id'] ?? 0);
        
        if ($coin_symbol === '' || $amount <= 0) {
            $error = 'Lütfen geçerli bir coin ve miktar girin.';
        } else {
            try {
                $db->beginTransaction();
                
                $stmt = $db->prepare("SELECT id, balance, available FROM wallets WHERE user_id = ? AND coin_symbol = ? LIMIT 1");
                $stmt->execute([$user_id, $coin_symbol]);
                $wallet = $stmt->fetch();
                
                if (!$wallet) {
                    throw new Exception('Cüzdan bulunamadı.');
                }
                
                if ($wallet['available'] < $amount) {
                    throw new Exception('Yetersiz bakiye. Mevcut: ' . $wallet['available'] . ' ' . $coin_symbol);
                }
                
                // Bakiyeyi düşür
                $upd = $db->prepare("UPDATE wallets SET balance = balance - ?, available = available - ? WHERE id = ?");
                $upd->execute([$amount, $amount, $wallet['id']]);
                
                // Negatif tutar için withdraw işlemi kaydı
                $price = 1.0;
                if (!empty($crypto_prices_post[$coin_symbol]['raw_price'])) {
                    $price = (float)$crypto_prices_post[$coin_symbol]['raw_price'];
                } elseif ($coin_symbol === 'USDT') {
                    $price = 1.0;
                }
                
                $tx = $db->prepare("INSERT INTO transactions (user_id, type, coin_symbol, amount, price, status, created_at) VALUES (?, 'withdraw', ?, ?, ?, 'completed', NOW())");
                $tx->execute([$user_id, $coin_symbol, -$amount, $price]);
                
                $db->commit();
                
                $_SESSION['admin_flash_success'] = 'Bakiye düşürüldü: ' . $amount . ' ' . $coin_symbol;
                header('Location: user_detail.php?id=' . $user_id);
                exit;
            } catch (Exception $e) {
                if ($db->inTransaction()) { $db->rollBack(); }
                $error = 'Bakiye düşürülemedi: ' . $e->getMessage();
                error_log("Balance reduce error: " . $e->getMessage());
            }
        }
    }
    
    // Bakiye sıfırlama (admin)
    if (isset($_POST['action']) && $_POST['action'] === 'reset_balance') {
        $coin_symbol = strtoupper(trim($_POST['coin_symbol'] ?? ''));
        $wallet_id = (int)($_POST['wallet_id'] ?? 0);
        
        if ($coin_symbol === '' || $wallet_id <= 0) {
            $error = 'Geçersiz işlem.';
        } else {
            try {
                $db->beginTransaction();
                
                $stmt = $db->prepare("SELECT id, balance, available FROM wallets WHERE id = ? AND user_id = ? AND coin_symbol = ? LIMIT 1");
                $stmt->execute([$wallet_id, $user_id, $coin_symbol]);
                $wallet = $stmt->fetch();
                
                if (!$wallet) {
                    throw new Exception('Cüzdan bulunamadı.');
                }
                
                $old_balance = $wallet['balance'];
                
                if ($old_balance > 0) {
                    // Bakiyeyi sıfırla
                    $upd = $db->prepare("UPDATE wallets SET balance = 0, available = 0 WHERE id = ?");
                    $upd->execute([$wallet['id']]);
                    
                    // İşlem kaydı
                    $price = 1.0;
                    if (!empty($crypto_prices_post[$coin_symbol]['raw_price'])) {
                        $price = (float)$crypto_prices_post[$coin_symbol]['raw_price'];
                    } elseif ($coin_symbol === 'USDT') {
                        $price = 1.0;
                    }
                    
                    $tx = $db->prepare("INSERT INTO transactions (user_id, type, coin_symbol, amount, price, status, created_at) VALUES (?, 'withdraw', ?, ?, ?, 'completed', NOW())");
                    $tx->execute([$user_id, $coin_symbol, -$old_balance, $price]);
                }
                
                $db->commit();
                
                $_SESSION['admin_flash_success'] = 'Bakiye sıfırlandı: ' . $old_balance . ' ' . $coin_symbol;
                header('Location: user_detail.php?id=' . $user_id);
                exit;
            } catch (Exception $e) {
                if ($db->inTransaction()) { $db->rollBack(); }
                $error = 'Bakiye sıfırlanamadı: ' . $e->getMessage();
                error_log("Balance reset error: " . $e->getMessage());
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kullanıcı Detayı - Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="icon" type="image/x-icon" href="/favicon.ico">
</head>
<body class="bg-gray-50">
    <div class="min-h-screen flex">
        <!-- Sidebar -->
        <div class="w-64 bg-white shadow-lg">
            <div class="p-6">
                <h2 class="text-xl font-bold text-gray-800">Admin Panel</h2>
                <p class="text-sm text-gray-600"><?php echo htmlspecialchars($adminEmail); ?></p>
            </div>
            <nav class="mt-6">
                <a href="index.php" class="flex items-center px-6 py-3 text-gray-600 hover:bg-gray-100">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2H5a2 2 0 00-2-2z"></path>
                    </svg>
                    Dashboard
                </a>
                <a href="users.php" class="flex items-center px-6 py-3 bg-blue-50 text-blue-600 border-r-2 border-blue-600">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197m13.5-9a2.5 2.5 0 11-5 0 2.5 2.5 0 015 0z"></path>
                    </svg>
                    Kullanıcılar
                </a>
                <a href="coins.php" class="flex items-center px-6 py-3 text-gray-600 hover:bg-gray-100">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"></path>
                    </svg>
                    Coinler
                </a>
                <a href="logout.php" class="flex items-center px-6 py-3 text-red-600 hover:bg-red-50">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"></path>
                    </svg>
                    Çıkış Yap
                </a>
            </nav>
        </div>

        <!-- Main Content -->
        <div class="flex-1 p-8">
            <!-- Header -->
            <div class="mb-8">
                <div class="flex items-center justify-between">
                    <div>
                        <h1 class="text-3xl font-bold text-gray-800">Kullanıcı Detayı</h1>
                        <p class="text-gray-600 mt-2"><?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?></p>
                    </div>
                    <a href="users.php" class="px-4 py-2 bg-gray-300 text-gray-700 rounded-md hover:bg-gray-400 focus:outline-none focus:ring-2 focus:ring-gray-500">
                        ← Geri Dön
                    </a>
                </div>
            </div>

            <!-- Success/Error Messages -->
            <?php if (isset($success)): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>

            <?php if (isset($error)): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>

            <!-- User Info -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-8">
                <!-- User Details -->
                <div class="bg-white rounded-lg shadow p-6">
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Kullanıcı Bilgileri</h3>
                    <div class="space-y-4">
                        <div class="flex items-center">
                            <div class="flex-shrink-0 h-16 w-16">
                                <div class="h-16 w-16 rounded-full bg-gray-300 flex items-center justify-center">
                                    <span class="text-xl font-medium text-gray-700">
                                        <?php echo strtoupper(substr($user['first_name'], 0, 1) . substr($user['last_name'], 0, 1)); ?>
                                    </span>
                                </div>
                            </div>
                            <div class="ml-4">
                                <div class="text-lg font-medium text-gray-900">
                                    <?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?>
                                    <?php if ($user['is_admin']): ?>
                                        <span class="ml-2 inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-purple-100 text-purple-800">
                                            Admin
                                        </span>
                                    <?php endif; ?>
                                </div>
                                <div class="text-sm text-gray-500">ID: <?php echo $user['id']; ?></div>
                            </div>
                        </div>
                        
                        <div class="border-t pt-4">
                            <dl class="space-y-2">
                                <div>
                                    <dt class="text-sm font-medium text-gray-500">Email</dt>
                                    <dd class="text-sm text-gray-900"><?php echo htmlspecialchars($user['email']); ?></dd>
                                </div>
                                <div>
                                    <dt class="text-sm font-medium text-gray-500">Telefon</dt>
                                    <dd class="text-sm text-gray-900"><?php echo htmlspecialchars($user['phone'] ?? 'Belirtilmemiş'); ?></dd>
                                </div>
                                <div>
                                    <dt class="text-sm font-medium text-gray-500">Kayıt Tarihi</dt>
                                    <dd class="text-sm text-gray-900"><?php echo date('d.m.Y H:i', strtotime($user['created_at'])); ?></dd>
                                </div>
                                <div>
                                    <dt class="text-sm font-medium text-gray-500">Son Giriş</dt>
                                    <dd class="text-sm text-gray-900">
                                        <?php echo $user['last_login'] ? date('d.m.Y H:i', strtotime($user['last_login'])) : 'Hiç giriş yapmamış'; ?>
                                    </dd>
                                </div>
                                <div>
                                    <dt class="text-sm font-medium text-gray-500">Durum</dt>
                                    <dd class="text-sm">
                                        <form method="POST" class="inline">
                                            <input type="hidden" name="action" value="toggle_status">
                                            <input type="hidden" name="new_status" value="<?php echo $user['is_active'] ? 0 : 1; ?>">
                                            <button type="submit" class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo $user['is_active'] ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?> hover:opacity-80">
                                                <?php echo $user['is_active'] ? 'Aktif' : 'Pasif'; ?>
                                            </button>
                                        </form>
                                    </dd>
                                </div>
                            </dl>
                        </div>
                    </div>
                </div>

                <!-- Balance Summary -->
                <div class="bg-white rounded-lg shadow p-6">
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Bakiye Özeti</h3>
                    <div class="space-y-4">
                        <div class="text-center">
                            <div class="text-3xl font-bold text-gray-900">$<?php echo number_format($total_balance, 2); ?></div>
                            <div class="text-sm text-gray-500">Toplam Bakiye (USD)</div>
                        </div>
                        
                        <!-- TRY Balance Card -->
                        <?php
                        $try_wallet = null;
                        foreach ($wallets as $wallet) {
                            if ($wallet['coin_symbol'] === 'TRY') {
                                $try_wallet = $wallet;
                                break;
                            }
                        }
                        ?>
                        <?php if ($try_wallet): ?>
                        <div class="bg-gradient-to-r from-red-50 to-red-100 rounded-lg p-4 border border-red-200">
                            <div class="flex items-center justify-between">
                                <div class="flex items-center">
                                    <div class="w-8 h-8 bg-red-500 rounded-full flex items-center justify-center mr-3">
                                        <span class="text-white font-bold text-sm">₺</span>
                                    </div>
                                    <div>
                                        <div class="text-sm font-medium text-gray-700">Türk Lirası</div>
                                        <div class="text-xs text-gray-500">TRY</div>
                                    </div>
                                </div>
                                <div class="text-right">
                                    <div class="text-lg font-bold text-gray-900">₺<?php echo formatBalance($try_wallet['balance'], 'TRY'); ?></div>
                                    <div class="text-xs text-gray-500">Mevcut</div>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>
                        
                        <div class="space-y-2">
                            <div class="flex justify-between">
                                <span class="text-sm text-gray-500">Cüzdan Sayısı</span>
                                <span class="text-sm font-medium text-gray-900"><?php echo count($wallets); ?></span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-sm text-gray-500">Aktif Cüzdan</span>
                                <span class="text-sm font-medium text-gray-900">
                                    <?php echo count(array_filter($wallets, function($w) { return $w['balance'] > 0; })); ?>
                                </span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-sm text-gray-500">Toplam İşlem</span>
                                <span class="text-sm font-medium text-gray-900"><?php echo count($transactions); ?></span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Quick Actions -->
                <div class="bg-white rounded-lg shadow p-6">
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Hızlı İşlemler</h3>
                    <div class="space-y-3">
                        <a href="../trade.php?user_id=<?php echo $user['id']; ?>" 
                           class="w-full block text-center px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                            İşlem Geçmişi
                        </a>
                        <a href="../wallet.php?user_id=<?php echo $user['id']; ?>" 
                           class="w-full block text-center px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700">
                            Cüzdan Detayı
                        </a>
                        <a href="../portfolio.php?user_id=<?php echo $user['id']; ?>" 
                           class="w-full block text-center px-4 py-2 bg-purple-600 text-white rounded-md hover:bg-purple-700">
                            Portföy Görünümü
                        </a>

                        <!-- Bakiye Yönetimi -->
                        <div class="pt-3 border-t mt-3">
                            <h4 class="text-sm font-medium text-gray-800 mb-3">Bakiye Yönetimi</h4>
                            
                            <!-- Bakiye Ekleme -->
                            <div class="mb-4">
                                <form method="POST" action="" class="space-y-2" onsubmit="return confirm('Bakiye eklemek istediğinizden emin misiniz?');">
                                    <input type="hidden" name="action" value="add_balance">
                                    <label class="block text-xs font-medium text-gray-700 mb-1">Bakiye Ekle</label>
                                    <div class="flex gap-2">
                                        <select name="coin_symbol" class="flex-1 px-3 py-2 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500" required>
                                            <option value="">Para birimi</option>
                                            <option value="TRY">TRY</option>
                                            <option value="USDT">USDT</option>
                                            <option value="BTC">BTC</option>
                                            <option value="ETH">ETH</option>
                                            <option value="SOL">SOL</option>
                                            <option value="BNB">BNB</option>
                                        </select>
                                        <input type="number" name="amount" step="0.00000001" min="0.00000001" placeholder="Miktar" 
                                               class="flex-1 px-3 py-2 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500" required>
                                    </div>
                                    <button type="submit" class="w-full px-4 py-2 bg-green-600 text-white text-sm rounded-md hover:bg-green-700 transition-colors">
                                        ➕ Bakiye Ekle
                                    </button>
                                </form>
                            </div>
                            
                            <!-- Bakiye Düşürme -->
                            <div class="mb-4">
                                <form method="POST" action="" class="space-y-2" onsubmit="return confirm('Bakiye düşürmek istediğinizden emin misiniz?');">
                                    <input type="hidden" name="action" value="reduce_balance">
                                    <label class="block text-xs font-medium text-gray-700 mb-1">Bakiye Düşür</label>
                                    <div class="flex gap-2">
                                        <select name="coin_symbol" id="reduce_coin" class="flex-1 px-3 py-2 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-red-500" required>
                                            <option value="">Para birimi</option>
                                            <?php foreach ($wallets as $wallet): ?>
                                                <?php if ($wallet['available'] > 0): ?>
                                                    <option value="<?php echo $wallet['coin_symbol']; ?>" data-wallet-id="<?php echo $wallet['id']; ?>" data-available="<?php echo $wallet['available']; ?>">
                                                        <?php echo $wallet['coin_symbol']; ?> (Mevcut: <?php echo formatBalance($wallet['available'], $wallet['coin_symbol']); ?>)
                                                    </option>
                                                <?php endif; ?>
                                            <?php endforeach; ?>
                                        </select>
                                        <input type="number" name="amount" id="reduce_amount" step="0.00000001" min="0.00000001" placeholder="Miktar" 
                                               class="flex-1 px-3 py-2 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-red-500" required>
                                        <input type="hidden" name="wallet_id" id="reduce_wallet_id">
                                    </div>
                                    <button type="submit" class="w-full px-4 py-2 bg-red-600 text-white text-sm rounded-md hover:bg-red-700 transition-colors">
                                        ➖ Bakiye Düşür
                                    </button>
                                </form>
                            </div>
                            
                            <!-- Bakiye Sıfırlama -->
                            <div>
                                <form method="POST" action="" class="space-y-2" onsubmit="return confirm('Bu cüzdanın bakiyesini tamamen sıfırlamak istediğinizden emin misiniz? Bu işlem geri alınamaz!');">
                                    <input type="hidden" name="action" value="reset_balance">
                                    <label class="block text-xs font-medium text-gray-700 mb-1">Bakiye Sıfırla</label>
                                    <div class="flex gap-2">
                                        <select name="coin_symbol" id="reset_coin" class="flex-1 px-3 py-2 text-sm border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-orange-500" required>
                                            <option value="">Para birimi</option>
                                            <?php foreach ($wallets as $wallet): ?>
                                                <?php if ($wallet['balance'] > 0): ?>
                                                    <option value="<?php echo $wallet['coin_symbol']; ?>" data-wallet-id="<?php echo $wallet['id']; ?>" data-balance="<?php echo $wallet['balance']; ?>">
                                                        <?php echo $wallet['coin_symbol']; ?> (Bakiye: <?php echo formatBalance($wallet['balance'], $wallet['coin_symbol']); ?>)
                                                    </option>
                                                <?php endif; ?>
                                            <?php endforeach; ?>
                                        </select>
                                        <input type="hidden" name="wallet_id" id="reset_wallet_id">
                                    </div>
                                    <button type="submit" class="w-full px-4 py-2 bg-orange-600 text-white text-sm rounded-md hover:bg-orange-700 transition-colors">
                                        🔄 Bakiye Sıfırla
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Wallets -->
            <div class="bg-white rounded-lg shadow mb-6">
                <div class="px-6 py-4 border-b border-gray-200">
                    <h3 class="text-lg font-medium text-gray-900">Cüzdanlar</h3>
                </div>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Coin</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Bakiye</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Müsait</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Dondurulmuş</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Değer (USDT)</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">İşlemler</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php if (empty($wallets)): ?>
                                <tr>
                                    <td colspan="6" class="px-6 py-4 text-center text-gray-500">
                                        Cüzdan bulunamadı
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($wallets as $wallet): 
                                    $coin_symbol = $wallet['coin_symbol'];
                                    $price = 1;
                                    
                                    if (isset($crypto_prices[$coin_symbol])) {
                                        $price = floatval($crypto_prices[$coin_symbol]['raw_price']);
                                    }
                                    
                                    $value = $wallet['balance'] * $price;
                                ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="flex items-center">
                                            <div class="flex-shrink-0 h-8 w-8">
                                                <div class="h-8 w-8 rounded-full bg-gray-300 flex items-center justify-center">
                                                    <span class="text-xs font-medium text-gray-700">
                                                        <?php echo strtoupper(substr($coin_symbol, 0, 2)); ?>
                                                    </span>
                                                </div>
                                            </div>
                                            <div class="ml-3">
                                                <div class="text-sm font-medium text-gray-900"><?php echo $coin_symbol; ?></div>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 font-mono">
                                        <?php echo formatBalance($wallet['balance'], $wallet['coin_symbol']); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 font-mono">
                                        <?php echo formatBalance($wallet['available'], $wallet['coin_symbol']); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 font-mono">
                                        <?php echo formatBalance($wallet['frozen'], $wallet['coin_symbol']); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 font-semibold">
                                        $<?php echo number_format($value, 2); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm">
                                        <div class="flex items-center gap-2">
                                            <?php if ($wallet['available'] > 0): ?>
                                                <button onclick="quickReduce('<?php echo $wallet['coin_symbol']; ?>', <?php echo $wallet['id']; ?>, <?php echo $wallet['available']; ?>)" 
                                                        class="px-3 py-1 bg-red-100 text-red-700 rounded text-xs hover:bg-red-200 transition-colors">
                                                    ➖ Düşür
                                                </button>
                                            <?php endif; ?>
                                            <?php if ($wallet['balance'] > 0): ?>
                                                <form method="POST" class="inline" onsubmit="return confirm('Bu cüzdanın bakiyesini tamamen sıfırlamak istediğinizden emin misiniz?');">
                                                    <input type="hidden" name="action" value="reset_balance">
                                                    <input type="hidden" name="coin_symbol" value="<?php echo $coin_symbol; ?>">
                                                    <input type="hidden" name="wallet_id" value="<?php echo $wallet['id']; ?>">
                                                    <button type="submit" class="px-3 py-1 bg-orange-100 text-orange-700 rounded text-xs hover:bg-orange-200 transition-colors">
                                                        🔄 Sıfırla
                                                    </button>
                                                </form>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Recent Transactions -->
            <div class="bg-white rounded-lg shadow">
                <div class="px-6 py-4 border-b border-gray-200">
                    <h3 class="text-lg font-medium text-gray-900">Son İşlemler</h3>
                </div>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tür</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Coin</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Miktar</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Fiyat</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Durum</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tarih</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php if (empty($transactions)): ?>
                                <tr>
                                    <td colspan="6" class="px-6 py-4 text-center text-gray-500">
                                        İşlem bulunamadı
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($transactions as $tx): 
                                    $type_map = [
                                        'deposit' => ['name' => 'Yatırma', 'color' => 'green'],
                                        'withdraw' => ['name' => 'Çekme', 'color' => 'red'],
                                        'transfer' => ['name' => 'Transfer', 'color' => 'blue'],
                                        'trade' => ['name' => floatval($tx['amount']) > 0 ? 'Alış' : 'Satış', 'color' => floatval($tx['amount']) > 0 ? 'green' : 'red'],
                                        'staking' => ['name' => 'Stake', 'color' => 'yellow'],
                                        'reward' => ['name' => 'Ödül', 'color' => 'green']
                                    ];
                                    $type_info = $type_map[$tx['type']] ?? ['name' => ucfirst($tx['type']), 'color' => 'gray'];
                                ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-<?php echo $type_info['color']; ?>-100 text-<?php echo $type_info['color']; ?>-800">
                                            <?php echo $type_info['name']; ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        <?php echo $tx['coin_symbol']; ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 font-mono">
                                        <?php echo formatBalance($tx['amount'], $tx['coin_symbol']); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        $<?php echo number_format($tx['price'], 2); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                            <?php echo ucfirst($tx['status']); ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        <?php echo date('d.m.Y H:i', strtotime($tx['created_at'])); ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Bakiye düşürme için select değişikliğinde wallet_id güncelle
        document.getElementById('reduce_coin')?.addEventListener('change', function() {
            const selected = this.options[this.selectedIndex];
            const walletId = selected.dataset.walletId;
            const available = selected.dataset.available;
            
            if (walletId) {
                document.getElementById('reduce_wallet_id').value = walletId;
                document.getElementById('reduce_amount').max = available;
            }
        });

        // Bakiye sıfırlama için select değişikliğinde wallet_id güncelle
        document.getElementById('reset_coin')?.addEventListener('change', function() {
            const selected = this.options[this.selectedIndex];
            const walletId = selected.dataset.walletId;
            
            if (walletId) {
                document.getElementById('reset_wallet_id').value = walletId;
            }
        });

        // Hızlı bakiye düşürme fonksiyonu
        function quickReduce(coin, walletId, available) {
            const amount = prompt(`Düşürmek istediğiniz miktarı girin (Maksimum: ${available} ${coin}):`, '');
            
            if (amount && parseFloat(amount) > 0) {
                if (parseFloat(amount) > available) {
                    alert('Yetersiz bakiye! Maksimum: ' + available + ' ' + coin);
                    return;
                }
                
                if (confirm(`${amount} ${coin} düşürmek istediğinizden emin misiniz?`)) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <input type="hidden" name="action" value="reduce_balance">
                        <input type="hidden" name="coin_symbol" value="${coin}">
                        <input type="hidden" name="amount" value="${amount}">
                        <input type="hidden" name="wallet_id" value="${walletId}">
                    `;
                    document.body.appendChild(form);
                    form.submit();
                }
            }
        }
    </script>
</body>
</html>
