<?php
session_start();

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$adminEmail = $_SESSION['admin_email'] ?? 'admin';

require_once '../config/database.php';

// Veritabanından istatistikleri çek
try {
    $db = getDB();
    
    // Toplam kullanıcı sayısı
    $stmt = $db->prepare("SELECT COUNT(*) as total FROM users");
    $stmt->execute();
    $totalUsers = $stmt->fetch()['total'] ?? 0;
    
    // Toplam para yatırma (deposits tablosu)
    $totalDeposits = 0;
    try {
        $stmt = $db->prepare("SELECT SUM(amount) as total FROM deposits WHERE status = 'completed'");
        $stmt->execute();
        $totalDeposits = $stmt->fetch()['total'] ?? 0;
    } catch (Exception $e) {
        // Tablo yoksa veya hata varsa 0
    }
    
    // Toplam para çekme (withdrawals tablosu)
    $totalWithdrawals = 0;
    try {
        $stmt = $db->prepare("SELECT SUM(amount) as total FROM withdrawals WHERE status = 'completed'");
        $stmt->execute();
        $totalWithdrawals = $stmt->fetch()['total'] ?? 0;
    } catch (Exception $e) {
        // Tablo yoksa veya hata varsa 0
    }
    
    // Toplam işlem hacmi (sadece trade işlemleri için amount * price toplamı)
    $totalVolume = 0;
    try {
        // Önce trade işlemlerini hesapla
        $stmt = $db->prepare("
            SELECT COALESCE(SUM(ABS(amount) * COALESCE(price, 1))), 0) as total 
            FROM transactions 
            WHERE status = 'completed' AND type = 'trade'
        ");
        $stmt->execute();
        $totalVolume = $stmt->fetch()['total'] ?? 0;
        
        // Eğer trade hacmi yoksa, deposits + withdrawals toplamını göster
        if ($totalVolume == 0) {
            $stmt = $db->prepare("
                SELECT COALESCE(SUM(ABS(amount) * COALESCE(price, 1))), 0) as total 
                FROM transactions 
                WHERE status = 'completed' AND type IN ('deposit', 'withdraw')
            ");
            $stmt->execute();
            $totalVolume = $stmt->fetch()['total'] ?? 0;
        }
    } catch (Exception $e) {
        // Tablo yoksa veya hata varsa 0
        $totalVolume = 0;
    }
    
    // Son işlem - deposits, withdrawals ve transactions tablolarından en son olanı bul
    $lastDeposit = null;
    $lastWithdrawal = null;
    $lastTransactionRecord = null;
    
    // Deposits tablosundan son işlem
    try {
        $stmt = $db->prepare("
            SELECT d.id, d.user_id, d.amount, d.status, d.created_at, u.first_name, u.last_name, u.email, 'deposit' as type
            FROM deposits d
            LEFT JOIN users u ON d.user_id = u.id
            ORDER BY d.created_at DESC
            LIMIT 1
        ");
        $stmt->execute();
        $lastDeposit = $stmt->fetch();
    } catch (Exception $e) {
        // Tablo yoksa sessizce geç
    }
    
    // Withdrawals tablosundan son işlem
    try {
        $stmt = $db->prepare("
            SELECT w.id, w.user_id, w.amount, w.status, w.created_at, u.first_name, u.last_name, u.email, 'withdrawal' as type
            FROM withdrawals w
            LEFT JOIN users u ON w.user_id = u.id
            ORDER BY w.created_at DESC
            LIMIT 1
        ");
        $stmt->execute();
        $lastWithdrawal = $stmt->fetch();
    } catch (Exception $e) {
        // Tablo yoksa sessizce geç
    }
    
    // Transactions tablosundan son işlem
    try {
        $stmt = $db->prepare("
            SELECT t.id, t.user_id, t.amount, t.status, t.created_at, u.first_name, u.last_name, u.email, t.type
            FROM transactions t
            LEFT JOIN users u ON t.user_id = u.id
            WHERE t.type IN ('trade', 'deposit', 'withdraw', 'transfer')
            ORDER BY t.created_at DESC
            LIMIT 1
        ");
        $stmt->execute();
        $lastTransactionRecord = $stmt->fetch();
    } catch (Exception $e) {
        // Tablo yoksa sessizce geç
    }
    
    // En son işlemi bul (tüm kaynaklardan karşılaştır)
    $lastTransaction = null;
    $candidates = [];
    
    if ($lastDeposit) {
        $candidates[] = $lastDeposit;
    }
    if ($lastWithdrawal) {
        $candidates[] = $lastWithdrawal;
    }
    if ($lastTransactionRecord) {
        $candidates[] = $lastTransactionRecord;
    }
    
    if (!empty($candidates)) {
        usort($candidates, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        $lastTransaction = $candidates[0];
    }
    
    // Aktif kullanıcılar (son 30 gün içinde giriş yapan)
    // Önce last_login kolonunun varlığını kontrol et
    $activeUsers = 0;
    try {
        // last_login kolonu var mı kontrol et
        $checkCol = $db->prepare("SHOW COLUMNS FROM users LIKE 'last_login'");
        $checkCol->execute();
        $hasLastLogin = $checkCol->fetch();
        
        if ($hasLastLogin) {
            $stmt = $db->prepare("SELECT COUNT(DISTINCT id) as active FROM users WHERE last_login IS NOT NULL AND last_login >= DATE_SUB(NOW(), INTERVAL 30 DAY)");
            $stmt->execute();
            $activeUsers = $stmt->fetch()['active'] ?? 0;
        } else {
            // last_login kolonu yoksa, updated_at kullan (son 30 gün içinde güncellenen)
            $stmt = $db->prepare("SELECT COUNT(DISTINCT id) as active FROM users WHERE updated_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)");
            $stmt->execute();
            $activeUsers = $stmt->fetch()['active'] ?? 0;
        }
    } catch (Exception $e) {
        $activeUsers = 0;
    }
    
    // Bekleyen işlemler
    $pendingDeposits = 0;
    $pendingWithdrawals = 0;
    
    try {
        $stmt = $db->prepare("SELECT COUNT(*) as pending FROM deposits WHERE status = 'pending'");
        $stmt->execute();
        $pendingDeposits = $stmt->fetch()['pending'] ?? 0;
    } catch (Exception $e) {
        // Tablo yoksa 0
    }
    
    try {
        $stmt = $db->prepare("SELECT COUNT(*) as pending FROM withdrawals WHERE status = 'pending'");
        $stmt->execute();
        $pendingWithdrawals = $stmt->fetch()['pending'] ?? 0;
    } catch (Exception $e) {
        // Tablo yoksa 0
    }
    
} catch (Exception $e) {
    $totalUsers = 0;
    $totalDeposits = 0;
    $totalWithdrawals = 0;
    $totalVolume = 0;
    $lastTransaction = null;
    $activeUsers = 0;
    $pendingDeposits = 0;
    $pendingWithdrawals = 0;
}

// Format helper
function formatMoney($amount) {
    return number_format($amount, 2, ',', '.') . ' ₺';
}

function formatDate($date) {
    if (!$date) return '-';
    $timestamp = strtotime($date);
    $months = [
        1 => 'Ocak', 2 => 'Şubat', 3 => 'Mart', 4 => 'Nisan',
        5 => 'Mayıs', 6 => 'Haziran', 7 => 'Temmuz', 8 => 'Ağustos',
        9 => 'Eylül', 10 => 'Ekim', 11 => 'Kasım', 12 => 'Aralık'
    ];
    return date('d', $timestamp) . ' ' . $months[(int)date('n', $timestamp)] . ' ' . date('Y H:i', $timestamp);
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body>
<div class="flex min-h-screen bg-muted/20">
    <!-- Sidebar -->
    <div class="h-full fixed max-lg:w-[calc(100vw-35px)] top-0 left-0 z-50 flex-col bg-gradient-to-b from-background to-muted/20 border-r hidden lg:block w-64">
        <div class="h-14 flex items-center px-4 border-b">
            <a class="flex items-center gap-2" href="/">
                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-gauge h-6 w-6 text-primary"><path d="m12 14 4-4"></path><path d="M3.34 19a10 10 0 1 1 17.32 0"></path></svg>
                <span class="font-semibold text-lg">Coin Platform</span>
            </a>
        </div>
        <div dir="ltr" class="max-h-[calc(100vh-100px)] overflow-auto" style="position: relative; --radix-scroll-area-corner-width: 0px; --radix-scroll-area-corner-height: 0px;">
            <div class="flex-1 overflow-auto py-4">
                <div class="py-2">
                    <div class="px-4 mb-2"><h2 class="text-xs font-semibold text-muted-foreground uppercase tracking-wider">Platform</h2></div>
                    <div class="space-y-1 px-2">
                        <a class="block" href="index.php">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 bg-primary/10 text-primary shadow-sm">
                                <span class="mr-3 flex-shrink-0 text-primary">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-layout-dashboard h-4 w-4"><rect width="7" height="9" x="3" y="3" rx="1"></rect><rect width="7" height="5" x="14" y="3" rx="1"></rect><rect width="7" height="9" x="14" y="12" rx="1"></rect><rect width="7" height="5" x="3" y="16" rx="1"></rect></svg>
                                </span>
                                <span class="truncate">Genel Bakış</span>
                                <span class="ml-auto h-1.5 w-1.5 rounded-full bg-primary"></span>
                            </div>
                        </a>
                        <a class="block" href="coins.php">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-coins h-4 w-4"><circle cx="8" cy="8" r="6"></circle><path d="M18.09 10.37A6 6 0 1 1 10.34 18"></path><path d="M7 6h1v4"></path><path d="m16.71 13.88.7.71-2.82 2.82"></path></svg>
                                </span>
                                <span class="truncate">Coinler</span>
                            </div>
                        </a>
                    </div>
                </div>

                <div class="py-2 mt-2">
                    <div class="px-4 mb-2"><h2 class="text-xs font-semibold text-muted-foreground uppercase tracking-wider">Kullanıcılar ve İşlemler</h2></div>
                    <div class="space-y-1 px-2">
                        <a class="block" href="users.php">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-users h-4 w-4"><path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"></path><circle cx="9" cy="7" r="4"></circle><path d="M22 21v-2a4 4 0 0 0-3-3.87"></path><path d="M16 3.13a4 4 0 0 1 0 7.75"></path></svg>
                                </span>
                                <span class="truncate">Kullanıcılar</span>
                            </div>
                        </a>
                        <a class="block" href="#transactions">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-credit-card h-4 w-4"><rect width="20" height="14" x="2" y="5" rx="2"></rect><line x1="2" x2="22" y1="10" y2="10"></line></svg>
                                </span>
                                <span class="truncate">İşlemler</span>
                            </div>
                        </a>
                        <a class="block" href="#orders">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-line-chart h-4 w-4"><path d="M3 3v18h18"></path><path d="m19 9-5 5-4-4-3 3"></path></svg>
                                </span>
                                <span class="truncate">Emir Geçmişi</span>
                            </div>
                        </a>
                        <a class="block" href="#trades">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-line-chart h-4 w-4"><path d="M3 3v18h18"></path><path d="m19 9-5 5-4-4-3 3"></path></svg>
                                </span>
                                <span class="truncate">Trading</span>
                            </div>
                        </a>
                    </div>
                </div>

                <div class="py-2 mt-2">
                    <div class="px-4 mb-2"><h2 class="text-xs font-semibold text-muted-foreground uppercase tracking-wider">İçerik Yönetimi</h2></div>
                    <div class="space-y-1 px-2">
                        <a class="block" href="#contact-messages">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-message-square h-4 w-4"><path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path></svg>
                                </span>
                                <span class="truncate">İletişim Mesajları</span>
                            </div>
                        </a>
                        <a class="block" href="#promotions">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-gift h-4 w-4"><rect x="3" y="8" width="18" height="4" rx="1"></rect><path d="M12 8v13"></path><path d="M19 12v7a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2v-7"></path><path d="M7.5 8a2.5 2.5 0 0 1 0-5A4.8 8 0 0 1 12 8a4.8 8 0 0 1 4.5-5 a2.5 2.5 0 0 1 0 5"></path></svg>
                                </span>
                                <span class="truncate">Promosyonlar</span>
                            </div>
                        </a>
                        <a class="block" href="bank_accounts.php">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-building2 h-4 w-4"><path d="M6 22V4a2 2 0 0 1 2-2h8a2 2 0 0 1 2 2v18Z"></path><path d="M6 12H4a2 2 0 0 0-2 2v6a2 2 0 0 0 2 2h2"></path><path d="M18 9h2a2 2 0 0 1 2 2v9a2 2 0 0 1-2 2h-2"></path><path d="M10 6h4"></path><path d="M10 10h4"></path><path d="M10 14h4"></path><path d="M10 18h4"></path></svg>
                                </span>
                                <span class="truncate">Banka Hesapları</span>
                            </div>
                        </a>
                        <a class="block" href="crypto_addresses.php">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-coins h-4 w-4"><circle cx="8" cy="8" r="6"></circle><path d="M18.09 10.37A6 6 0 1 1 10.34 18"></path><path d="M7 6h1v4"></path><path d="m16.71 13.88.7.71-2.82 2.82"></path></svg>
                                </span>
                                <span class="truncate">Kripto Adresleri</span>
                            </div>
                        </a>
                        <a class="block" href="deposits.php">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-credit-card h-4 w-4"><rect width="20" height="14" x="2" y="5" rx="2"></rect><line x1="2" x2="22" y1="10" y2="10"></line></svg>
                                </span>
                                <span class="truncate">Para Yatırma</span>
                            </div>
                        </a>
                        <a class="block" href="withdrawals.php">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-banknote h-4 w-4"><rect width="20" height="12" x="2" y="6" rx="2"></rect><circle cx="12" cy="12" r="2"></circle><path d="M6 12h.01M18 12h.01"></path></svg>
                                </span>
                                <span class="truncate">Çekim Talepleri</span>
                            </div>
                        </a>
                        <a class="block" href="components.php">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-layout h-4 w-4"><rect width="18" height="18" x="3" y="3" rx="2" ry="2"></rect><line x1="3" x2="21" y1="9" y2="9"></line><line x1="9" x2="9" y1="21" y2="9"></line></svg>
                                </span>
                                <span class="truncate">Header & Footer</span>
                            </div>
                        </a>
                        <a class="block" href="legal_pages.php">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-file-text h-4 w-4"><path d="M15 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V7Z"></path><path d="M14 2v4a2 2 0 0 0 2 2h4"></path><path d="M10 9H8"></path><path d="M16 13H8"></path><path d="M16 17H8"></path></svg>
                                </span>
                                <span class="truncate">Yasal Sayfalar</span>
                            </div>
                        </a>
                    </div>
                </div>

                <div class="py-2 mt-2">
                    <div class="px-4 mb-2"><h2 class="text-xs font-semibold text-muted-foreground uppercase tracking-wider">Sistem</h2></div>
                    <div class="space-y-1 px-2">
                        <a class="block" href="#settings">
                            <div class="group flex items-center py-2.5 px-3 text-sm font-medium rounded-lg transition-all duration-200 text-muted-foreground hover:bg-muted/50 hover:text-foreground">
                                <span class="mr-3 flex-shrink-0 text-muted-foreground group-hover:text-foreground">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-settings h-4 w-4"><path d="M12.22 2h-.44a2 2 0 0 0-2 2v.18a2 2 0 0 1-1 1.73l-.43.25a2 2 0 0 1-2 0l-.15-.08a2 2 0 0 0-2.73.73l-.22.38a2 2 0 0 0 .73 2.73l.15.1a2 2 0 0 1 1 1.72v.51a2 2 0 0 1-1 1.74l-.15.09a2 2 0 0 0-.73 2.73l.22.38a2 2 0 0 0 2.73.73l.15-.08a2 2 0 0 1 2 0l.43.25a2 2 0 0 1 1 1.73V20a2 2 0 0 0 2 2h.44a2 2 0 0 0 2-2v-.18a2 2 0 0 1 1-1.73l.43-.25a2 2 0 0 1 2 0l.15.08a2 2 0 0 0 2.73-.73l.22-.39a2 2 0 0 0-.73-2.73l-.15-.08a2 2 0 0 1-1-1.74v-.5a2 2 0 0 1 1-1.74l.15-.09a2 2 0 0 0 .73-2.73l-.22-.38a2 2 0 0 0-2.73-.73l-.15.08a2 2 0 0 1-2 0l-.43-.25a2 2 0 0 1-1-1.73V4a2 2 0 0 0-2-2z"></path><circle cx="12" cy="12" r="3"></circle></svg>
                                </span>
                                <span class="truncate">Ayarlar</span>
                            </div>
                        </a>
                    </div>
                </div>
            </div>
        </div>
        <div class="p-1 border-t">
            <a href="logout.php" class="flex w-full items-center px-3 py-2.5 text-sm font-medium rounded-lg text-red-600 hover:bg-red-50/50 transition-colors">
                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-log-out mr-3 h-4 w-4"><path d="M9 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h4"></path><polyline points="16 17 21 12 16 7"></polyline><line x1="21" x2="9" y1="12" y2="12"></line></svg>
                <span>Çıkış Yap</span>
            </a>
        </div>
    </div>

    <!-- Main -->
    <div class="flex-1 flex flex-col">
        <header class="sticky top-0 z-40 w-full border-b bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60">
            <div class="flex h-14 items-center">
                <div class="flex items-center gap-2 lg:hidden pl-6">
                    <button class="inline-flex items-center justify-center gap-2 whitespace-nowrap rounded-md text-sm font-medium hover:bg-accent hover:text-accent-foreground h-9 w-9 md:hidden" type="button" aria-haspopup="dialog" aria-expanded="false" aria-controls="radix-:r0:" data-state="closed">
                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-menu h-5 w-5"><line x1="4" x2="20" y1="12" y2="12"></line><line x1="4" x2="20" y1="6" y2="6"></line><line x1="4" x2="20" y1="18" y2="18"></line></svg>
                        <span class="sr-only">Toggle Menu</span>
                    </button>
                </div>
                <div class="flex flex-1 items-center justify-between space-x-2 md:justify-end">
                    <div class="w-full flex-1 md:w-auto md:flex-none"></div>
                    <nav class="flex items-center gap-2 pr-6">
                        <button class="inline-flex items-center justify-center gap-2 whitespace-nowrap text-sm font-medium hover:bg-accent hover:text-accent-foreground px-4 py-2 relative h-9 w-9 rounded-full" type="button" id="radix-:r3:" aria-haspopup="menu" aria-expanded="false" data-state="closed">
                            <span class="relative flex shrink-0 overflow-hidden rounded-full h-9 w-9 border-2 border-primary/10">
                                <span class="flex h-full w-full items-center justify-center rounded-full bg-primary/5 text-primary">A</span>
                            </span>
                            <span class="sr-only">Kullanıcı menüsü</span>
                        </button>
                    </nav>
                </div>
            </div>
        </header>
        <main class="flex-1 ml-[255px] max-lg:ml-0">
            <div class="py-6">
                <div class="mx-auto max-w-7xl">
                    <div class="p-6 md:p-8">
                        <div class="flex-1 space-y-6">
                            <!-- Header -->
                            <div class="flex items-center justify-between">
                                <div>
                                    <h2 class="text-3xl font-bold tracking-tight text-gray-900">Dashboard</h2>
                                    <p class="text-sm text-gray-500 mt-1">Platform genel bakış ve istatistikler</p>
                                </div>
                                <div class="text-sm text-gray-500">
                                    <?php 
                                    $months_tr = [
                                        'January' => 'Ocak', 'February' => 'Şubat', 'March' => 'Mart', 'April' => 'Nisan',
                                        'May' => 'Mayıs', 'June' => 'Haziran', 'July' => 'Temmuz', 'August' => 'Ağustos',
                                        'September' => 'Eylül', 'October' => 'Ekim', 'November' => 'Kasım', 'December' => 'Aralık'
                                    ];
                                    $dateStr = date('d F Y, H:i');
                                    foreach ($months_tr as $en => $tr) {
                                        $dateStr = str_replace($en, $tr, $dateStr);
                                    }
                                    echo $dateStr;
                                    ?>
                                </div>
                            </div>

                            <!-- İstatistik Kartları -->
                            <div class="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
                                <!-- Toplam Kullanıcı -->
                                <div class="rounded-xl border border-gray-200 bg-white shadow-sm hover:shadow-md transition-shadow overflow-hidden">
                                    <div class="p-6">
                                        <div class="flex items-center justify-between">
                                            <div>
                                                <p class="text-sm font-medium text-gray-600 mb-1">Toplam Kullanıcı</p>
                                                <p class="text-3xl font-bold text-gray-900"><?php echo number_format($totalUsers); ?></p>
                                                <p class="text-xs text-gray-500 mt-2"><?php echo $activeUsers; ?> aktif (30 gün)</p>
                                            </div>
                                            <div class="p-3 bg-blue-100 rounded-xl">
                                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="h-6 w-6 text-blue-600">
                                                    <path stroke-linecap="round" stroke-linejoin="round" d="M15 19.128a9.38 9.38 0 0 0 2.625.372 9.337 9.337 0 0 0 4.121-.952 4.125 4.125 0 0 0-7.533-2.493M15 19.128v-.003c0-1.113-.285-2.16-.786-3.07M15 19.128v.106A12.318 12.318 0 0 1 8.624 21c-2.331 0-4.512-.645-6.374-1.766l-.001-.109a6.375 6.375 0 0 1 11.964-3.07M12 6.375a3.375 3.375 0 1 1-6.75 0 3.375 3.375 0 0 1 6.75 0Zm8.25 2.25a2.625 2.625 0 1 1-5.25 0 2.625 2.625 0 0 1 5.25 0Z"></path>
                                                </svg>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Toplam Para Yatırma -->
                                <div class="rounded-xl border border-gray-200 bg-gradient-to-br from-green-50 to-white shadow-sm hover:shadow-md transition-shadow overflow-hidden">
                                    <div class="p-6">
                                        <div class="flex items-center justify-between">
                                            <div>
                                                <p class="text-sm font-medium text-gray-600 mb-1">Toplam Yatırma</p>
                                                <p class="text-3xl font-bold text-green-600"><?php echo formatMoney($totalDeposits); ?></p>
                                                <p class="text-xs text-gray-500 mt-2"><?php echo $pendingDeposits; ?> bekleyen</p>
                                            </div>
                                            <div class="p-3 bg-green-100 rounded-xl">
                                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="h-6 w-6 text-green-600">
                                                    <path stroke-linecap="round" stroke-linejoin="round" d="M2.25 18 9 11.25l4.306 4.306a11.95 11.95 0 0 1 5.814-5.518l2.74-1.22m0 0-5.94-2.281m5.94 2.28-2.28 5.941"></path>
                                                </svg>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Toplam Para Çekme -->
                                <div class="rounded-xl border border-gray-200 bg-gradient-to-br from-red-50 to-white shadow-sm hover:shadow-md transition-shadow overflow-hidden">
                                    <div class="p-6">
                                        <div class="flex items-center justify-between">
                                            <div>
                                                <p class="text-sm font-medium text-gray-600 mb-1">Toplam Çekim</p>
                                                <p class="text-3xl font-bold text-red-600"><?php echo formatMoney($totalWithdrawals); ?></p>
                                                <p class="text-xs text-gray-500 mt-2"><?php echo $pendingWithdrawals; ?> bekleyen</p>
                                            </div>
                                            <div class="p-3 bg-red-100 rounded-xl">
                                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="h-6 w-6 text-red-600">
                                                    <path stroke-linecap="round" stroke-linejoin="round" d="M21 12a2.25 2.25 0 0 0-2.25-2.25H15a3 3 0 1 1-6 0H5.25A2.25 2.25 0 0 0 3 12m18 0v6a2.25 2.25 0 0 1-2.25 2.25H5.25A2.25 2.25 0 0 1 3 18v-6m18 0V9M3 12V9m18 0a2.25 2.25 0 0 0-2.25-2.25H5.25A2.25 2.25 0 0 0 3 9m18 0V6a2.25 2.25 0 0 0-2.25-2.25H5.25A2.25 2.25 0 0 0 3 6v3"></path>
                                                </svg>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Toplam İşlem Hacmi -->
                                <div class="rounded-xl border border-gray-200 bg-gradient-to-br from-purple-50 to-white shadow-sm hover:shadow-md transition-shadow overflow-hidden">
                                    <div class="p-6">
                                        <div class="flex items-center justify-between">
                                            <div>
                                                <p class="text-sm font-medium text-gray-600 mb-1">İşlem Hacmi</p>
                                                <p class="text-3xl font-bold text-purple-600"><?php echo formatMoney($totalVolume); ?></p>
                                                <p class="text-xs text-gray-500 mt-2">Tamamlanan işlemler</p>
                                            </div>
                                            <div class="p-3 bg-purple-100 rounded-xl">
                                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="h-6 w-6 text-purple-600">
                                                    <path stroke-linecap="round" stroke-linejoin="round" d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6"></path>
                                                </svg>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Son İşlem ve Hızlı Aksiyonlar -->
                            <div class="grid gap-6 lg:grid-cols-2">
                                <!-- Son İşlem -->
                                <div class="rounded-xl border border-gray-200 bg-white shadow-sm overflow-hidden">
                                    <div class="px-6 py-4 border-b border-gray-200 bg-gray-50">
                                        <h3 class="text-lg font-semibold text-gray-900 flex items-center gap-2">
                                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="h-5 w-5 text-gray-600">
                                                <path stroke-linecap="round" stroke-linejoin="round" d="M12 6v6h4.5m4.5 0a9 9 0 1 1-18 0 9 9 0 0 1 18 0Z"></path>
                                            </svg>
                                            Son İşlem
                                        </h3>
                                    </div>
                                    <div class="p-6">
                                        <?php if ($lastTransaction): ?>
                                            <div class="space-y-4">
                                                <div class="flex items-start justify-between">
                                                    <div class="flex-1">
                                                        <div class="flex items-center gap-3 mb-2">
                                                            <div class="p-2 rounded-lg <?php echo $lastTransaction['type'] === 'deposit' ? 'bg-green-100' : 'bg-red-100'; ?>">
                                                                <?php if ($lastTransaction['type'] === 'deposit'): ?>
                                                                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="h-5 w-5 text-green-600">
                                                                        <path stroke-linecap="round" stroke-linejoin="round" d="M12 6v12m6-6H6"></path>
                                                                    </svg>
                                                                <?php else: ?>
                                                                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="h-5 w-5 text-red-600">
                                                                        <path stroke-linecap="round" stroke-linejoin="round" d="M19.5 12h-15"></path>
                                                                    </svg>
                                                                <?php endif; ?>
                                                            </div>
                                                            <div>
                                                                <p class="font-semibold text-gray-900">
                                                                    <?php 
                                                                    $name = trim(($lastTransaction['first_name'] ?? '') . ' ' . ($lastTransaction['last_name'] ?? ''));
                                                                    echo htmlspecialchars($name ?: 'Bilinmeyen Kullanıcı');
                                                                    ?>
                                                                </p>
                                                                <p class="text-sm text-gray-500"><?php echo htmlspecialchars($lastTransaction['email'] ?? '-'); ?></p>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    <div class="text-right">
                                                        <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-semibold <?php echo $lastTransaction['status'] === 'completed' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'; ?>">
                                                            <?php echo ucfirst($lastTransaction['status']); ?>
                                                        </span>
                                                    </div>
                                                </div>
                                                
                                                <div class="grid grid-cols-2 gap-4 pt-4 border-t border-gray-100">
                                                    <div>
                                                        <p class="text-xs text-gray-500 mb-1">İşlem Tipi</p>
                                                        <p class="text-sm font-semibold text-gray-900">
                                                            <?php echo $lastTransaction['type'] === 'deposit' ? 'Para Yatırma' : 'Para Çekme'; ?>
                                                        </p>
                                                    </div>
                                                    <div>
                                                        <p class="text-xs text-gray-500 mb-1">Miktar</p>
                                                        <p class="text-sm font-semibold text-gray-900"><?php echo formatMoney($lastTransaction['amount']); ?></p>
                                                    </div>
                                                </div>
                                                
                                                <div class="pt-4 border-t border-gray-100">
                                                    <p class="text-xs text-gray-500 mb-1">Tarih</p>
                                                    <p class="text-sm font-medium text-gray-700"><?php echo formatDate($lastTransaction['created_at']); ?></p>
                                                </div>
                                            </div>
                                        <?php else: ?>
                                            <div class="text-center py-8">
                                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" class="h-12 w-12 text-gray-300 mx-auto mb-3">
                                                    <path stroke-linecap="round" stroke-linejoin="round" d="M12 9v6m3-3H9m12 0a9 9 0 1 1-18 0 9 9 0 0 1 18 0Z"></path>
                                                </svg>
                                                <p class="text-sm text-gray-500">Henüz işlem bulunmamaktadır</p>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>

                                <!-- Hızlı Aksiyonlar -->
                                <div class="rounded-xl border border-gray-200 bg-white shadow-sm overflow-hidden">
                                    <div class="px-6 py-4 border-b border-gray-200 bg-gray-50">
                                        <h3 class="text-lg font-semibold text-gray-900 flex items-center gap-2">
                                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="h-5 w-5 text-gray-600">
                                                <path stroke-linecap="round" stroke-linejoin="round" d="M3.75 13.5l10.5-11.25L12 10.5h8.25L9.75 21.75 12 13.5H3.75z"></path>
                                            </svg>
                                            Hızlı Aksiyonlar
                                        </h3>
                                    </div>
                                    <div class="p-6">
                                        <div class="grid grid-cols-2 gap-4">
                                            <a href="deposits.php" class="p-4 rounded-lg border border-gray-200 hover:border-green-300 hover:bg-green-50 transition-all group">
                                                <div class="flex items-center gap-3 mb-2">
                                                    <div class="p-2 bg-green-100 rounded-lg group-hover:bg-green-200 transition-colors">
                                                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="h-5 w-5 text-green-600">
                                                            <path stroke-linecap="round" stroke-linejoin="round" d="M2.25 18 9 11.25l4.306 4.306a11.95 11.95 0 0 1 5.814-5.518l2.74-1.22m0 0-5.94-2.281m5.94 2.28-2.28 5.941"></path>
                                                        </svg>
                                                    </div>
                                                    <div>
                                                        <p class="font-semibold text-gray-900 text-sm">Para Yatırma</p>
                                                        <p class="text-xs text-gray-500"><?php echo $pendingDeposits; ?> bekleyen</p>
                                                    </div>
                                                </div>
                                            </a>
                                            
                                            <a href="withdrawals.php" class="p-4 rounded-lg border border-gray-200 hover:border-red-300 hover:bg-red-50 transition-all group">
                                                <div class="flex items-center gap-3 mb-2">
                                                    <div class="p-2 bg-red-100 rounded-lg group-hover:bg-red-200 transition-colors">
                                                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="h-5 w-5 text-red-600">
                                                            <path stroke-linecap="round" stroke-linejoin="round" d="M21 12a2.25 2.25 0 0 0-2.25-2.25H15a3 3 0 1 1-6 0H5.25A2.25 2.25 0 0 0 3 12m18 0v6a2.25 2.25 0 0 1-2.25 2.25H5.25A2.25 2.25 0 0 1 3 18v-6m18 0V9M3 12V9m18 0a2.25 2.25 0 0 0-2.25-2.25H5.25A2.25 2.25 0 0 0 3 9m18 0V6a2.25 2.25 0 0 0-2.25-2.25H5.25A2.25 2.25 0 0 0 3 6v3"></path>
                                                        </svg>
                                                    </div>
                                                    <div>
                                                        <p class="font-semibold text-gray-900 text-sm">Para Çekme</p>
                                                        <p class="text-xs text-gray-500"><?php echo $pendingWithdrawals; ?> bekleyen</p>
                                                    </div>
                                                </div>
                                            </a>
                                            
                                            <a href="users.php" class="p-4 rounded-lg border border-gray-200 hover:border-blue-300 hover:bg-blue-50 transition-all group">
                                                <div class="flex items-center gap-3 mb-2">
                                                    <div class="p-2 bg-blue-100 rounded-lg group-hover:bg-blue-200 transition-colors">
                                                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="h-5 w-5 text-blue-600">
                                                            <path stroke-linecap="round" stroke-linejoin="round" d="M15 19.128a9.38 9.38 0 0 0 2.625.372 9.337 9.337 0 0 0 4.121-.952 4.125 4.125 0 0 0-7.533-2.493M15 19.128v-.003c0-1.113-.285-2.16-.786-3.07M15 19.128v.106A12.318 12.318 0 0 1 8.624 21c-2.331 0-4.512-.645-6.374-1.766l-.001-.109a6.375 6.375 0 0 1 11.964-3.07M12 6.375a3.375 3.375 0 1 1-6.75 0 3.375 3.375 0 0 1 6.75 0Zm8.25 2.25a2.625 2.625 0 1 1-5.25 0 2.625 2.625 0 0 1 5.25 0Z"></path>
                                                        </svg>
                                                    </div>
                                                    <div>
                                                        <p class="font-semibold text-gray-900 text-sm">Kullanıcılar</p>
                                                        <p class="text-xs text-gray-500"><?php echo number_format($totalUsers); ?> toplam</p>
                                                    </div>
                                                </div>
                                            </a>
                                            
                                            <a href="coins.php" class="p-4 rounded-lg border border-gray-200 hover:border-purple-300 hover:bg-purple-50 transition-all group">
                                                <div class="flex items-center gap-3 mb-2">
                                                    <div class="p-2 bg-purple-100 rounded-lg group-hover:bg-purple-200 transition-colors">
                                                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" class="h-5 w-5 text-purple-600">
                                                            <path stroke-linecap="round" stroke-linejoin="round" d="M12 6v12m-3-2.818.879.659c1.171.879 3.07.879 4.242 0 1.172-.879 1.172-2.303 0-3.182C13.536 12.219 12.768 12 12 12c-.725 0-1.45-.22-2.003-.659-1.106-.879-1.106-2.303 0-3.182s2.9-.879 4.006 0l.415.33M21 12a9 9 0 1 1-18 0 9 9 0 0 1 18 0Z"></path>
                                                        </svg>
                                                    </div>
                                                    <div>
                                                        <p class="font-semibold text-gray-900 text-sm">Coinler</p>
                                                        <p class="text-xs text-gray-500">Yönet</p>
                                                    </div>
                                                </div>
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>
</body>
</html>


